/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/


//
// This is a version of AmrDerive.cpp that calculates the sum of
// densities for each z value.
//

#include <new>
#include <iostream>
#include <cstdlib>
#include <cstring>

#include <unistd.h>

#include "REAL.H"
#include "Box.H"
#include "FArrayBox.H"
#include "ParmParse.H"
#include "ParallelDescriptor.H"
#include "DataServices.H"
#include "Utility.H"
#include "VisMF.H"
#include "Derived.H"


static
void
PrintUsage (char* progName)
{
    cout << "\nUsage:\n"
         << progName
         << "\n\tinfile"
         << "\n\n";
    exit(1);
}


int
main (int   argc,
      char* argv[])
{
    //
    // Make sure to catch new failures.
    //
    set_new_handler(BoxLib::OutOfMemory);

    if (argc == 1)
        PrintUsage(argv[0]);

    ParallelDescriptor::StartParallel(&argc, &argv);
    if(ParallelDescriptor::NProcs() > 1) {
      cerr << "Error:  serial only right now." << endl;
      BoxLib::Abort();
    }

    Real sumVal, firstVal, prevVal;

    FArrayBox::setFormat(FABio::FAB_IEEE_32);

   for(int iCDer(0); iCDer < 2; ++iCDer) {
    std::string currentDerived;
    if(iCDer == 0) {
      currentDerived = "density";
    } else {
      currentDerived = "tracer";
    }
     for(int iPlt(1); iPlt < argc; ++iPlt) {

      std::string iFile(argv[iPlt]);
      cout << "Summing " << currentDerived << " for " << iFile << endl;

      DataServices::SetBatchMode();
      FileType fileType(NEWPLT);
    
      DataServices dataServices(iFile, fileType);

      if ( ! dataServices.AmrDataOk()) {
        //
        // This calls ParallelDescriptor::EndParallel() and exit()
        //
        DataServices::Dispatch(DataServices::ExitRequest, NULL);
      }
      AmrData &amrData = dataServices.AmrDataRef();

      int finestFillLevel(0);  // just do level zero
      Box bProbDomainZero(amrData.ProbDomain()[0]);
      BoxArray baZSlabs(bProbDomainZero.length(ZDIR));
      for(int iSlab(0); iSlab < baZSlabs.size(); ++iSlab) {
        Box bTempSlab(bProbDomainZero);
        bTempSlab.setSmall(ZDIR, iSlab);
        bTempSlab.setBig(ZDIR, iSlab);
        baZSlabs.set(iSlab, bTempSlab);
      }

      MultiFab mfSumRho(baZSlabs, 1, 0);

      amrData.FillVar(mfSumRho, finestFillLevel, currentDerived);

      FArrayBox fabCheck(bProbDomainZero);
      amrData.FillVar(&fabCheck, fabCheck.box(), 0, currentDerived, 0);

      Array<Real> rSumRho(baZSlabs.size(), 0.0);
      Real dx(amrData.DxLevel()[finestFillLevel][XDIR]);
      Real dy(amrData.DxLevel()[finestFillLevel][YDIR]);
      for(int iSlab(0); iSlab < rSumRho.size(); ++iSlab) {
        rSumRho[iSlab] = mfSumRho[iSlab].sum(0) * dx * dy;
      }

      if (ParallelDescriptor::IOProcessor()) {
        Real sumCheck(0.0);
        std::string aSumFile(iFile);
        aSumFile += ".sum";
        aSumFile += currentDerived;
        ofstream outfile(aSumFile.c_str());
	outfile << endl << "# " << aSumFile << endl;
        for(int iSlab(rSumRho.size() - 1); iSlab >= 0; --iSlab) {
          outfile << (iSlab + 0.5) << "  " << (rSumRho[iSlab] / 10.0) << endl;
	  sumCheck += rSumRho[iSlab];
        }

	sumVal = fabCheck.sum(0);
        if(iPlt == 1) {
	  firstVal = sumVal;
	  prevVal  = sumVal;
        }

	outfile.precision(12);
        outfile << "# " << "sumCheck        = " << sumCheck << endl;
        outfile << "# " << "fabCheck * dxdy = " << sumVal * dx * dy << endl;
        outfile << "# " << "fabCheck.sum    = " << sumVal << endl;
        outfile << "# " << "fabCheck.delta0 = " << sumVal - firstVal << endl;
        outfile << "# " << "fabCheck.deltaP = " << sumVal - prevVal  << endl << endl;
	prevVal = sumVal;
        outfile.close();
      }

     }  // end for(iPlt...)
    } // end for(iCDer...)

    DataServices::Dispatch(DataServices::ExitRequest, NULL);
}
