/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file main.c
 * \brief Main entry functions
 */

#include <ffgtk.h>
#include <gio/gio.h>
#include <libgen.h>
#ifdef HAVE_DBUS
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#endif

void ffgtkInitLock( void );

/** Contains the program logo */
static GdkPixbuf *psLogo = NULL;
/** GTK main loop */
GMainLoop *psLoop = NULL;

/**
 * \brief Get ffgtk logo pixbuf
 * \return logo as gdkpixbuf
 */
GdkPixbuf *getLogo( void ) {
	return psLogo;
}

#ifdef HAVE_DBUS
/**
 * \brief DBus signal filter
 * \param psConnection dbus connection pointer
 * \param psMessage dbus message pointer
 * \param pUserData gtk main loop pointer
 * \return dbus result code
 */
static DBusHandlerResult dbusSignalFilter( DBusConnection *psConnection, DBusMessage *psMessage, void *pUserData ) {
	GMainLoop *psLoop = pUserData;

	if ( dbus_message_is_signal( psMessage, "org.freedesktop.Local", "Disconnected" ) ) {
		g_main_loop_quit( psLoop );
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if ( dbus_message_is_signal( psMessage, "org.tabos.ffgtk.dbus.Signal", "Call" ) ) {
		DBusError sError;
		gchar *pnStr = NULL;

		dbus_error_init( &sError );
		if ( dbus_message_get_args( psMessage, &sError, DBUS_TYPE_STRING, &pnStr, DBUS_TYPE_INVALID ) ) {
			g_print( "Call received: %s\n", pnStr );
			gchar *pnNumber = fullNumber( pnStr, FALSE );
			g_print( "Call received (processed): %s\n", pnNumber );
			dialNumberDialog( NULL, pnNumber );
			g_free( pnNumber );
		} else {
			g_print( "Call received, but error getting message: %s\n", sError.message );
			dbus_error_free( &sError );
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/**
 * \brief Init DBus interface
 * \param psLoop gtk main loop pointer
 * \return error code
 */
static int InitDbus( GMainLoop *psLoop ) {
	DBusConnection *psBus;
	DBusError sError;

	dbus_error_init( &sError );
	psBus = dbus_bus_get( DBUS_BUS_SESSION, &sError );
	if ( dbus_error_is_set( &sError ) ) {
		g_warning( "Failed to connect to the D-BUS daemon: %s", sError.message );
		dbus_error_free( &sError );
		return 1;
	}

	dbus_bus_add_match( psBus, "type='signal',interface='org.tabos.ffgtk.dbus.Signal'", NULL );
	dbus_connection_add_filter( psBus, dbusSignalFilter, psLoop, NULL );

	dbus_connection_setup_with_g_main( psBus, NULL );

	return 0;
}
#endif

#ifdef HAVE_FAXOPHONE
/**
 * \brief Spooler callback
 * \param psMonitor file monitor_dir
 * \param psFile file
 * \param psOtherFile other file
 * \param nEventType event type
 * \param pUserData user data
 */
static void spoolerCallback( GFileMonitor *psMonitor, GFile *psFile, GFile *psOtherFile, GFileMonitorEvent nEventType, gpointer pUserData ) {
	gchar *pnFile = g_file_get_path( psFile );
	const gchar *pnUserName = g_get_user_name();
	static int nType = -1;

	if ( pnFile == NULL || pnUserName == NULL ) {
		return;
	}

	Debug( KERN_DEBUG, "File: %s Type: %d (%s)  Event: %d (%s)\n", pnFile, 
				nType, GFileMonitorEvent2Text(nType), 
				nEventType, GFileMonitorEvent2Text( nEventType) );
	if ( nType == -1 && nEventType == G_FILE_MONITOR_EVENT_CREATED ) {
		nType = nEventType;
	}

	if ( nEventType == G_FILE_MONITOR_EVENT_CHANGES_DONE_HINT ) {
		if ( nType == G_FILE_MONITOR_EVENT_CREATED ) {
			nType = nEventType;
#ifndef G_OS_WIN32
			gchar *pnName = g_strdup_printf( "%s-", pnUserName );
			if ( has_file_extension( pnFile, ".tif" ) == FALSE && strncmp( basename( pnFile ), pnName, strlen( pnName ) ) == 0 ) {
#else
			if ( ( has_file_extension( pnFile, ".tif" ) == TRUE ) || ( has_file_extension( pnFile, ".tiff" ) == TRUE ) ) {
#endif
				Debug( KERN_DEBUG, _( "Print job received on spooler\n" ) );
				processFax( pnFile );
			}
#ifndef G_OS_WIN32
			g_free( pnName );
#endif
		}
		nType = -1;
	}

	g_free( pnFile );
}

/**
 * \brief Initialize new printer spool queue
 */
void InitSpoolWatcher( void ) {
	GFileMonitor *psMonitor = NULL;
	GFile *psFile = NULL;
	DIR *psDir = NULL;
#ifdef G_OS_WIN32
	gchar *pnDir = g_strdup_printf( "%s/spool/", getUserDir() );
#else
	gchar *pnDir = g_strdup_printf( "/var/spool/ffgtk" );
#endif

	/* Check if spooler is present, create directory if needed */
	psDir = opendir( pnDir );
	if ( psDir == NULL ) {
		g_mkdir( pnDir, 0777 );
	} else {
		closedir( psDir );
	}

	Debug( KERN_DEBUG, "Setting file monitor to '%s'\n", pnDir );

	/* Create GFile for GFileMonitor */
	psFile = g_file_new_for_path( pnDir );
	/* Create file monitor for spool directory */
	psMonitor = g_file_monitor_directory( psFile, 0, NULL, NULL);
	/* Set callback for file monitor */
	g_signal_connect( psMonitor, "changed", G_CALLBACK( spoolerCallback ), NULL );

	g_free( pnDir );
}
#endif

/**
 * \brief Main entry function.
 * Set window logo, localization, plugins, profiles, trayicon, server
 * \param argc argument count
 * \param argv argument vector
 * \return error code
 */
#ifdef G_OS_WIN32
int ffgtk_main(HINSTANCE hint, int argc, char **argv);
int ffgtk_main(HINSTANCE hint, int argc, char **argv) {
#else
int main( int argc, char **argv ) {
#endif
	/* Display version information */
	Debug( KERN_INFO, "ffgtk %s - %s\n", PACKAGE_VERSION, anSvnRevision );

	/* Initialize thread support */
	if ( !g_thread_supported() ) {
#if GLIB_CHECK_VERSION(2,31,0)
		/* Threads are already initialized */
#else
		g_thread_init( NULL );
#endif
#ifdef HAVE_DBUS
		dbus_g_thread_init();
#endif
	}
	gdk_threads_init();

	ffgtkInitLock();

	/* Initialize gtk */
	gtk_init( &argc, &argv );

	/* Set local bindings */
#ifdef G_OS_WIN32
	gchar *pnLocale = NULL;

	pnLocale = getDirectory( LOCALEDIR );
	bindtextdomain( GETTEXT_PACKAGE, pnLocale );
	g_free( pnLocale );

#else
	bindtextdomain( GETTEXT_PACKAGE, LOCALEDIR );
#endif
	bind_textdomain_codeset( GETTEXT_PACKAGE, "UTF-8" );
	textdomain( GETTEXT_PACKAGE );

	/* Create main loop, we will connect the dbus signal filter to this one */
	psLoop = g_main_loop_new( NULL, FALSE );

#ifdef HAVE_DBUS
	/* Init DBUS interface */
	InitDbus( psLoop );
#endif

	/* Setup program logo and set it as default window icon */
	gchar *pnLogoFile = g_build_filename( getDirectory( PKGDATADIR ), "callout.png", NULL );
	psLogo = gdk_pixbuf_new_from_file( pnLogoFile, NULL );
	g_free( pnLogoFile );
	gtk_window_set_default_icon( getLogo() );

	/* Create tray icon and connect menu and signals */
	TrayIconInit();
	
	/* Time to load the plugins */
#ifdef G_OS_WIN32
	gchar *pnPluginDir = getDirectory( "lib/ffgtk/bin" );
	Debug( KERN_DEBUG, "pnPluginDir: %s\n", pnPluginDir );
	PluginsLoad( pnPluginDir );
	g_free( pnPluginDir );
#else
	PluginsLoad( LIBDIR "plugins" );
#endif
	/* Time to load lookups */
	LookupLoad();
	/* Next step we load the profiles */
	ProfilesLoad();

	Debug( KERN_DEBUG, "ActionsLoad()\n" );
	/* Load actions */
	ActionsLoad();
	/* Load DTMF actions */
	DtmfActionsLoad();
	/* Load last calls */
	LoadLastcalls();
	/* Load diversity state */
	DiversityLoad();
	/* Start call-monitor port thread */
	InitMonitor();

#ifdef HAVE_FAXOPHONE

#ifdef __MACOSX__
	/* Start old cups printer waitqueue */
	PrintServerInit();
#else
	/* Initialize new fax spool watcher */
	InitSpoolWatcher();
#endif

#ifdef HAVE_DBUS
	InitBluetooth();
#endif

	/* Initialize faxophone */
	InitFaxPhone();
#endif

	recreateAppMenu();

	/* Notify complete startup */
	Debug( KERN_DEBUG, "startup complete\n" );
	gdk_notify_startup_complete();

	/* Finally give control to gtk main loop */
	Debug( KERN_DEBUG, "g_main_loop_run\n" );

	g_main_loop_run( psLoop );

	if ( getTrayIcon() ) {
		g_object_unref( G_OBJECT( getTrayIcon() ) );
	}

#ifdef HAVE_FAXOPHONE
	/* Close faxophone */
	Debug( KERN_DEBUG, "CloseFaxPhone()\n" );
	CloseFaxPhone();
	Debug( KERN_DEBUG, "CloseFaxPhone() done\n" );
#endif

	/* Logout */
	if ( getActiveProfile() ) {
	Debug( KERN_DEBUG, "routerLogout()\n" );
		routerLogout( getActiveProfile() );
	}

	/* Unload plugins */
	Debug( KERN_DEBUG, "PluginsUnload()\n" );
	PluginsUnload();
	Debug( KERN_DEBUG, "PluginsUnload() done\n" );

	if ( psLogo != NULL ) {
		Debug( KERN_DEBUG, "unref()\n" );
		g_object_unref( G_OBJECT( psLogo ) );
	}

	Debug( KERN_DEBUG, "all done\n" );
	/* all done, get out of here */
	return 0;
}
