/***************************************************************************
                 cnetaddr.h  -  dclib network address functions
                             -------------------
    begin                : Sat Aug 30 2008
    copyright            : (C) 2001-2003 by Mathias Küster
    copyright            : (C) 2008 by Edward Sheldrake
    email                : ejs1920@yahoo.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CNETADDR_H
#define CNETADDR_H

/**
 * @author Mathias Küster, Edward Sheldrake.
 *
 * IP address functions, mainly the DNS lookup, but also getting the
 * address from network interfaces and some utility functions.
 *
 * Moved out of CSocket and gethostbyname() was replaced with getaddrinfo().
 * Mutex not required by getaddrinfo().
 *
 * Most of the names got shortened and end with I4 to indicate they
 * work with IPv4 addresses, and most functions take an optional CString*
 * for returning any error message.
 *
 * On WIN32 this class won't work until CSocket::SysInit() has initialised
 * WinSock2, dclibInitDepLibs() handles that and any other libraries.
 */

#include <dclib/dcos.h>
#include <dclib/core/cstring.h>
#include <dclib/core/clist.h>

class CNetAddr {
public:
	/**
	 * Gets the list of network interfaces on this machine.
	 * If there is some problem and error is not null then
	 * if will be set to the error message.
	 *
	 * Returns the number of interfaces in the list.
	 */
	static long GetInterfaceList( CList<CString> * interfaceList, CString * error = 0 );
	
	/**
	 * Returns the IPv4 address for the network interface.
	 * If there was some problem an empty string is returned
	 * and if error is not null then it will be set to the error message.
	 */
	static CString GetInterfaceI4( CString hwif, CString * error = 0 );
	
	/**
	 * Puts the first IPv4 address for the host into sin.sin_addr and returns true.
	 * If lookup fails returns false and sets error if it is not null.
	 */
	static bool GetHostI4( const char * host, struct sockaddr_in * sin, CString * error = 0 );
	
	/**
	 * Returns the first IPv4 address for the host as a CString.
	 * If lookup fails returns an empty string and sets error if it is not null.
	 */
	static CString GetHostI4( CString host, CString * error = 0 );
	
	/**
	 * Splits host and port.
	 */
	static void ParseHost( CString host, CString & ip, unsigned int & port );
	
	/**
	 * Returns true if the IPv4 address (in dots and numbers format)
	 * is in one of the private address space blocks.
	 */
	static bool IsPrivateI4( const char * ip );
	
	/**
	 * Returns true if the ip address (in dots and numbers format)
	 * is a valid IPv4 address.
	 */
	static bool IsValidI4( const char * ip );
	
protected:
	/** Constructor - not created, all functions static */
	CNetAddr() {};
	/** Destructor - not destroyed, all functions static */
	~CNetAddr() {};
};

#endif // CNETADDR_H
