#define SIMDE_TEST_X86_AVX512_INSN rorv

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/rorv.h>

static int
test_simde_mm_rorv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[4];
    const int32_t b[4];
    const int32_t r[4];
  } test_vec[] = {
    { { -INT32_C(    69354328), -INT32_C(  1197998948),  INT32_C(   293111754),  INT32_C(  1832942147) },
      { -INT32_C(   411975625),  INT32_C(  1192461355),  INT32_C(  1780271933),  INT32_C(  1641329593) },
      { -INT32_C(  1149677065), -INT32_C(  1818815745), -INT32_C(  1950073264), -INT32_C(  1606606410) } },
    { {  INT32_C(   207401327),  INT32_C(  2009396397), -INT32_C(  1098367621), -INT32_C(   282277449) },
      { -INT32_C(  1261003383),  INT32_C(  1593567521),  INT32_C(  1858607028),  INT32_C(   986684618) },
      { -INT32_C(  1215943080), -INT32_C(  1142785450), -INT32_C(  2083013656),  INT32_C(  1845218098) } },
    { { -INT32_C(    96064435), -INT32_C(  1687090656), -INT32_C(    10880441),  INT32_C(  1290700227) },
      {  INT32_C(  1140900898),  INT32_C(  1638071213), -INT32_C(   573609198),  INT32_C(  1427611399) },
      {  INT32_C(  2123467539),  INT32_C(  1359272840),  INT32_C(  2123497430), -INT32_C(  2036736757) } },
    { { -INT32_C(   347120181), -INT32_C(  1350123161),  INT32_C(  2125389755), -INT32_C(  2016764827) },
      {  INT32_C(   231459424), -INT32_C(   680628539), -INT32_C(   525058343), -INT32_C(  1472869411) },
      { -INT32_C(   347120181),  INT32_C(  1031550475),  INT32_C(  1466949055),  INT32_C(  1045750572) } },
    { { -INT32_C(  1869380312),  INT32_C(    20912454),  INT32_C(  1568665080), -INT32_C(   370849399) },
      { -INT32_C(   654921709), -INT32_C(   139500514),  INT32_C(  2144887970),  INT32_C(  1478954287) },
      {  INT32_C(  1889866258),  INT32_C(    83649816),  INT32_C(   392166270), -INT32_C(  1827417142) } },
    { { -INT32_C(   655836526), -INT32_C(   858183725), -INT32_C(  1658234860), -INT32_C(  1266282847) },
      { -INT32_C(   577995585), -INT32_C(  2083177248), -INT32_C(   821908064),  INT32_C(  1294412474) },
      { -INT32_C(  1311673051), -INT32_C(   858183725), -INT32_C(  1658234860),  INT32_C(   562276461) } },
    { { -INT32_C(  1205530652),  INT32_C(  1250230070), -INT32_C(   102257321),  INT32_C(  2074963388) },
      { -INT32_C(   900187415),  INT32_C(   374156662), -INT32_C(  1780133926),  INT32_C(  1591872889) },
      { -INT32_C(   228846969),  INT32_C(   335337770),  INT32_C(  2045466110), -INT32_C(   692658627) } },
    { {  INT32_C(  1393952540),  INT32_C(  1587386886),  INT32_C(    56067399), -INT32_C(   595720974) },
      { -INT32_C(  1264134338), -INT32_C(   540347644), -INT32_C(  1133203390), -INT32_C(   652585283) },
      {  INT32_C(  1280842865),  INT32_C(  1709824416), -INT32_C(  1059724975), -INT32_C(   470800490) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m128i r = simde_mm_rorv_epi32(a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i a = simde_test_x86_random_i32x4();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m128i r = simde_mm_rorv_epi32(a, b);

    simde_test_x86_write_i32x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_mask_rorv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[4];
    const simde__mmask8 k;
    const int32_t a[4];
    const int32_t b[4];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT32_C(   161852646), -INT32_C(  1964798349),  INT32_C(  1283013027),  INT32_C(  1063349850) },
      UINT8_C(233),
      { -INT32_C(   609719213),  INT32_C(   995443397), -INT32_C(  1712393867),  INT32_C(  1082183316) },
      {  INT32_C(    45320564), -INT32_C(  1515897492),  INT32_C(   620753590), -INT32_C(  1492304301) },
      { -INT32_C(  2033893958), -INT32_C(  1964798349),  INT32_C(  1283013027),  INT32_C(   433227792) } },
    { {  INT32_C(  1870837162),  INT32_C(  1823135735),  INT32_C(  1543936200), -INT32_C(   610498969) },
      UINT8_C( 15),
      { -INT32_C(  1921196721),  INT32_C(  1950556546),  INT32_C(  1606969377),  INT32_C(   722104181) },
      { -INT32_C(   937264655), -INT32_C(  1131376860),  INT32_C(   455339413), -INT32_C(   651493751) },
      {  INT32_C(  1856489150),  INT32_C(   658780696),  INT32_C(  1128336126), -INT32_C(  1164606153) } },
    { {  INT32_C(  1583785691), -INT32_C(   372069944), -INT32_C(  2025285103), -INT32_C(    88976887) },
      UINT8_C(204),
      {  INT32_C(  1676722900),  INT32_C(  1090038867), -INT32_C(   825682992), -INT32_C(   458579394) },
      { -INT32_C(  1297348600),  INT32_C(  1975752154),  INT32_C(   813583070), -INT32_C(   771982852) },
      {  INT32_C(  1583785691), -INT32_C(   372069944),  INT32_C(   992235331),  INT32_C(  1252664302) } },
    { { -INT32_C(  1909134277),  INT32_C(  1775119769),  INT32_C(  2117572672),  INT32_C(  1130619194) },
      UINT8_C(233),
      { -INT32_C(  1513818865),  INT32_C(    58931640),  INT32_C(   822064055), -INT32_C(  1670590288) },
      {  INT32_C(   842398213),  INT32_C(  1634901705), -INT32_C(  1214516778),  INT32_C(  1671552596) },
      {  INT32_C(  2100176808),  INT32_C(  1775119769),  INT32_C(  2117572672), -INT32_C(   854914618) } },
    { { -INT32_C(  1945606701),  INT32_C(  1435470750), -INT32_C(   276394433),  INT32_C(  1686893151) },
      UINT8_C(236),
      {  INT32_C(  1605736129), -INT32_C(   113961464), -INT32_C(  1873941070), -INT32_C(   211504754) },
      {  INT32_C(  1167192249),  INT32_C(   226813567),  INT32_C(  1584165740), -INT32_C(  1085550338) },
      { -INT32_C(  1945606701),  INT32_C(  1435470750), -INT32_C(   618068770), -INT32_C(   846019013) } },
    { {  INT32_C(  1864302694), -INT32_C(   915909609), -INT32_C(   816204223),  INT32_C(   549633383) },
      UINT8_C(173),
      {  INT32_C(   975988051),  INT32_C(  1554397929),  INT32_C(  2002453926),  INT32_C(  1356667472) },
      { -INT32_C(  1905832903),  INT32_C(  1808740533),  INT32_C(  1204985738), -INT32_C(  1242172831) },
      {  INT32_C(   372418973), -INT32_C(   915909609),  INT32_C(  1771951809),  INT32_C(   678333736) } },
    { {  INT32_C(  1106190680),  INT32_C(    43947355), -INT32_C(   361104998),  INT32_C(  1295734291) },
      UINT8_C(163),
      { -INT32_C(   749151326),  INT32_C(  1230881706), -INT32_C(  2018794347), -INT32_C(  1142988647) },
      { -INT32_C(   468311729), -INT32_C(  1199630145), -INT32_C(   389322351),  INT32_C(  1200298148) },
      { -INT32_C(  1220172111), -INT32_C(  1833203884), -INT32_C(   361104998),  INT32_C(  1295734291) } },
    { { -INT32_C(  1642404877),  INT32_C(  1005025190), -INT32_C(  1245539813),  INT32_C(  1097900786) },
      UINT8_C(195),
      { -INT32_C(  1618860410),  INT32_C(   238041765),  INT32_C(   515053573), -INT32_C(  2045642077) },
      { -INT32_C(  1960005868),  INT32_C(   698771351), -INT32_C(   870556886),  INT32_C(  1385127371) },
      {  INT32_C(   577268216),  INT32_C(  1618299420), -INT32_C(  1245539813),  INT32_C(  1097900786) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi32(test_vec[i].src);
    simde__m128i a = simde_x_mm_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m128i r = simde_mm_mask_rorv_epi32(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i src = simde_test_x86_random_i32x4();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i32x4();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m128i r = simde_mm_mask_rorv_epi32(src, k, a, b);

    simde_test_x86_write_i32x4(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_maskz_rorv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int32_t a[4];
    const int32_t b[4];
    const int32_t r[4];
  } test_vec[] = {
    { UINT8_C(131),
      {  INT32_C(  1260974353),  INT32_C(   978335521), -INT32_C(   455250198),  INT32_C(   804807808) },
      { -INT32_C(   121207920), -INT32_C(  2061307863),  INT32_C(  1766911500), -INT32_C(  1897028995) },
      { -INT32_C(   250524888), -INT32_C(  1868748773),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(147),
      {  INT32_C(  1286986005), -INT32_C(  1741230295),  INT32_C(   806886092), -INT32_C(  1782495470) },
      { -INT32_C(    20989682), -INT32_C(   922073893),  INT32_C(   927363988),  INT32_C(  1993004128) },
      {  INT32_C(  1683305175),  INT32_C(   115205427),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(173),
      {  INT32_C(  1859568255),  INT32_C(   322662137), -INT32_C(   836342906), -INT32_C(   438518996) },
      { -INT32_C(  1094657414),  INT32_C(  1465092580), -INT32_C(  1548186929), -INT32_C(   732942764) },
      { -INT32_C(  1246715941),  INT32_C(           0), -INT32_C(   686973876), -INT32_C(   877474211) } },
    { UINT8_C(240),
      { -INT32_C(  1779875289), -INT32_C(   383976067), -INT32_C(   568989149), -INT32_C(  1604715578) },
      {  INT32_C(  1166284731), -INT32_C(   199959702), -INT32_C(  1052133229),  INT32_C(   800201992) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C( 95),
      { -INT32_C(  1730296421), -INT32_C(   876886303), -INT32_C(   695100709),  INT32_C(   177353203) },
      { -INT32_C(  1871325259), -INT32_C(  1557894677), -INT32_C(   810752590),  INT32_C(   858774423) },
      { -INT32_C(   299049786), -INT32_C(   600213640),  INT32_C(  1723266468),  INT32_C(   610526741) } },
    { UINT8_C(160),
      { -INT32_C(   763245812),  INT32_C(   565005447),  INT32_C(   236225757), -INT32_C(   305979882) },
      { -INT32_C(    36154221),  INT32_C(  1571781752),  INT32_C(    66420520), -INT32_C(  1163646802) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(243),
      {  INT32_C(  1920633893), -INT32_C(  1118856391), -INT32_C(   825008465),  INT32_C(  1969340449) },
      {  INT32_C(   367877785), -INT32_C(  1925362930),  INT32_C(  1748713536),  INT32_C(   173798884) },
      {  INT32_C(  1028002489),  INT32_C(  1827075390),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(129),
      {  INT32_C(  1908047062),  INT32_C(   690059467),  INT32_C(   206303051), -INT32_C(  1348091824) },
      { -INT32_C(   105006418),  INT32_C(   926501623),  INT32_C(  2048696709), -INT32_C(   755292420) },
      { -INT32_C(   212220183),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_rorv_epi32(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i32x4();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m128i r = simde_mm_maskz_rorv_epi32(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x4(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_rorv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[8];
    const int32_t b[8];
    const int32_t r[8];
  } test_vec[] = {
    { {  INT32_C(   977838916), -INT32_C(  1754965992), -INT32_C(   522854253),  INT32_C(  2096133933),  INT32_C(  1162987586),  INT32_C(  1027930881),  INT32_C(  1330913981), -INT32_C(   542701157) },
      { -INT32_C(   988155987), -INT32_C(   614695096), -INT32_C(  1933889185), -INT32_C(   368530520),  INT32_C(  1966037364),  INT32_C(   313750613),  INT32_C(   828442518), -INT32_C(  1122957296) },
      {  INT32_C(   438424133),  INT32_C(   412575064), -INT32_C(  1045708505),  INT32_C(   763162743),  INT32_C(   478422101),  INT32_C(   668469738),  INT32_C(  1351284029),  INT32_C(   161210279) } },
    { {  INT32_C(  1082272504),  INT32_C(   152821417), -INT32_C(  1214917105), -INT32_C(   157114750),  INT32_C(  1282134519), -INT32_C(   597811643),  INT32_C(   906870821), -INT32_C(  1057808696) },
      { -INT32_C(   234850743),  INT32_C(  1694178132),  INT32_C(  1947963890),  INT32_C(   644529455), -INT32_C(   730606193),  INT32_C(   431018484),  INT32_C(  1515175569),  INT32_C(   622478300) },
      {  INT32_C(  2082488597), -INT32_C(  1108701039),  INT32_C(  1971580389),  INT32_C(  1023798597), -INT32_C(  1544644393), -INT32_C(   505127483), -INT32_C(   535651578),  INT32_C(   254930060) } },
    { {  INT32_C(   219683769),  INT32_C(   678499126), -INT32_C(   744714844), -INT32_C(   637925814), -INT32_C(   793874980), -INT32_C(   790012354), -INT32_C(   114673380),  INT32_C(   891176060) },
      { -INT32_C(  1774045601), -INT32_C(   289491894), -INT32_C(  1950196927),  INT32_C(  1030077537),  INT32_C(  1728910121), -INT32_C(  1908934799), -INT32_C(  1400413904),  INT32_C(    98674086) },
      {  INT32_C(   439367538), -INT32_C(   846586812),  INT32_C(  1775126226),  INT32_C(  1828520741), -INT32_C(   295151818), -INT32_C(  1356896140),  INT32_C(   958200106), -INT32_C(   254510831) } },
    { {  INT32_C(   647701468),  INT32_C(   403987159),  INT32_C(   379901621), -INT32_C(  1152120430), -INT32_C(  1927061220), -INT32_C(  2011473320),  INT32_C(  1647616700),  INT32_C(   610735432) },
      {  INT32_C(   273285945),  INT32_C(   304701021), -INT32_C(   953627339), -INT32_C(   226263850),  INT32_C(   914335454), -INT32_C(  1128293632), -INT32_C(  2061569219),  INT32_C(  1118406153) },
      {  INT32_C(  1301409299), -INT32_C(  1063070024),  INT32_C(   649439413),  INT32_C(  1344686829),  INT32_C(   881689714), -INT32_C(  2011473320),  INT32_C(   296031715), -INT32_C(  1542311030) } },
    { { -INT32_C(   430771319), -INT32_C(  2013758638),  INT32_C(   508436552),  INT32_C(  2064699805),  INT32_C(  2024902775),  INT32_C(  1748267051),  INT32_C(  1844335460),  INT32_C(  1655674841) },
      { -INT32_C(   582483573), -INT32_C(   983285635), -INT32_C(    35343776), -INT32_C(    75959164), -INT32_C(  1351406972), -INT32_C(    31938406), -INT32_C(   731183365),  INT32_C(   674634397) },
      {  INT32_C(  1899809374),  INT32_C(  1069800084),  INT32_C(   508436552), -INT32_C(   676262631),  INT32_C(  2005604615),  INT32_C(   219941594), -INT32_C(  1110807411),  INT32_C(   360496843) } },
    { { -INT32_C(  1727693284),  INT32_C(   509503934), -INT32_C(  1608760804), -INT32_C(  1147431881),  INT32_C(  1500188606), -INT32_C(  1302887497),  INT32_C(   646366089), -INT32_C(   112214819) },
      { -INT32_C(   124627910), -INT32_C(   636030530),  INT32_C(  1786393139), -INT32_C(  2061101625), -INT32_C(   606170844), -INT32_C(  1651689964), -INT32_C(   691858439),  INT32_C(   164631247) },
      {  INT32_C(  1096779558),  INT32_C(  2038015736), -INT32_C(  2008837117),  INT32_C(  1870083880), -INT32_C(   443109125),  INT32_C(  2017164069),  INT32_C(  1130480787),  INT32_C(  2042360479) } },
    { {  INT32_C(   620847719), -INT32_C(  2013325229),  INT32_C(   301037898), -INT32_C(  1281943666), -INT32_C(  1131514456), -INT32_C(  1537663829), -INT32_C(     8708817), -INT32_C(  1777841361) },
      {  INT32_C(    29035181),  INT32_C(  1820899874), -INT32_C(  1031898829),  INT32_C(   963974545),  INT32_C(   922092426),  INT32_C(  1339707167), -INT32_C(  1672587924),  INT32_C(  1311921824) },
      {  INT32_C(   322512907), -INT32_C(   503331308),  INT32_C(   791233086), -INT32_C(  1949869621),  INT32_C(  1781474205),  INT32_C(  1219639639), -INT32_C(   754976847), -INT32_C(  1777841361) } },
    { { -INT32_C(  2108690848), -INT32_C(   588261464), -INT32_C(   509711024),  INT32_C(   203101058),  INT32_C(   910299414), -INT32_C(   846914208),  INT32_C(   325702514), -INT32_C(  1973314775) },
      {  INT32_C(   839692425), -INT32_C(   686883961), -INT32_C(   356930456), -INT32_C(   705178433),  INT32_C(  1158363621), -INT32_C(   921530282), -INT32_C(  1914930333), -INT32_C(  1609089769) },
      {  INT32_C(   809576439),  INT32_C(  1371135919),  INT32_C(  1356963437),  INT32_C(   406202116), -INT32_C(  1313730424),  INT32_C(   343245622),  INT32_C(  1114454638), -INT32_C(  1019849964) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_rorv_epi32(a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i a = simde_test_x86_random_i32x8();
    simde__m256i b = simde_test_x86_random_i32x8();
    simde__m256i r = simde_mm256_rorv_epi32(a, b);

    simde_test_x86_write_i32x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_mask_rorv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[8];
    const simde__mmask8 k;
    const int32_t a[8];
    const int32_t b[8];
    const int32_t r[8];
  } test_vec[] = {
    { {  INT32_C(  1992212670), -INT32_C(  1795293304),  INT32_C(  1402052118), -INT32_C(   240989426), -INT32_C(   943632715), -INT32_C(  1882549633), -INT32_C(  1641953554), -INT32_C(   181191625) },
      UINT8_C(252),
      { -INT32_C(   259757071),  INT32_C(   101128552),  INT32_C(  1964268202),  INT32_C(  1244268284),  INT32_C(  1506406855), -INT32_C(  2109253444), -INT32_C(  1229331078),  INT32_C(   179482136) },
      { -INT32_C(  2097531367), -INT32_C(   108527281),  INT32_C(  1466867035),  INT32_C(  1789040803),  INT32_C(  1187212170),  INT32_C(  1053297348),  INT32_C(   133464815),  INT32_C(  1242670640) },
      {  INT32_C(  1992212670), -INT32_C(  1795293304), -INT32_C(  1567926962), -INT32_C(  1991950113),  INT32_C(  1909879420), -INT32_C(   937134709), -INT32_C(   889885325), -INT32_C(  1374156110) } },
    { {  INT32_C(   751570140),  INT32_C(  1764054029), -INT32_C(  1799318287), -INT32_C(  1224842708), -INT32_C(  1828929074), -INT32_C(  1160722997),  INT32_C(  2009252935),  INT32_C(  1187107690) },
      UINT8_C(223),
      { -INT32_C(   487755123),  INT32_C(   752047768),  INT32_C(  2035836950),  INT32_C(   675745638), -INT32_C(   789325557),  INT32_C(  1830268585),  INT32_C(  1138200432), -INT32_C(   568124080) },
      {  INT32_C(   683675792),  INT32_C(  2085917542),  INT32_C(  1660267771), -INT32_C(   947241797), -INT32_C(  1097368299), -INT32_C(  1691635925), -INT32_C(  1897987522), -INT32_C(  1335098848) },
      {  INT32_C(  1921901293),  INT32_C(  1622363482),  INT32_C(   722272975),  INT32_C(   149023941), -INT32_C(  1631035769), -INT32_C(  1160722997),  INT32_C(   257834433), -INT32_C(   568124080) } },
    { {  INT32_C(  2027432978), -INT32_C(  1141625664), -INT32_C(  1809978664),  INT32_C(  1012639526),  INT32_C(  1358623269), -INT32_C(   538171743),  INT32_C(  1215220520), -INT32_C(   537339187) },
      UINT8_C(  7),
      { -INT32_C(    54044720),  INT32_C(   903184971),  INT32_C(  1197238688), -INT32_C(  1234396988), -INT32_C(  1202143854),  INT32_C(  1960916905), -INT32_C(  2143213147), -INT32_C(   226025438) },
      { -INT32_C(  1024504201),  INT32_C(  1895351504), -INT32_C(   239578067),  INT32_C(  2124883180), -INT32_C(  1976041503), -INT32_C(   587327433),  INT32_C(  1666989889), -INT32_C(   698948769) },
      { -INT32_C(  1901092359), -INT32_C(  2109000235),  INT32_C(  1291991779),  INT32_C(  1012639526),  INT32_C(  1358623269), -INT32_C(   538171743),  INT32_C(  1215220520), -INT32_C(   537339187) } },
    { {  INT32_C(    43599153),  INT32_C(   913477897), -INT32_C(   785962267),  INT32_C(   810536527),  INT32_C(    96175822), -INT32_C(   522077794),  INT32_C(  1480802041),  INT32_C(  1395628321) },
      UINT8_C(222),
      {  INT32_C(  1508333000), -INT32_C(   230810169),  INT32_C(   323030852), -INT32_C(   438210210), -INT32_C(   427563476),  INT32_C(    98526151),  INT32_C(  1076311975),  INT32_C(   773814886) },
      { -INT32_C(  1769535793),  INT32_C(  1753793828),  INT32_C(   863750612),  INT32_C(  1729649723),  INT32_C(   172858178), -INT32_C(  1508954881), -INT32_C(   890882460),  INT32_C(  2147026352) },
      {  INT32_C(    43599153),  INT32_C(  2133058012),  INT32_C(   284442932), -INT32_C(  1137824804),  INT32_C(   966850955), -INT32_C(   522077794),  INT32_C(  1946317690),  INT32_C(  2053516831) } },
    { {  INT32_C(   806780940),  INT32_C(   446209606), -INT32_C(  1555229592), -INT32_C(  1291164304),  INT32_C(    12408833), -INT32_C(   391721852), -INT32_C(  1280078845), -INT32_C(  1640780654) },
      UINT8_C( 44),
      { -INT32_C(   411906487),  INT32_C(  2052033894),  INT32_C(  1089204954),  INT32_C(  1430363901),  INT32_C(   668549467),  INT32_C(  1932182247),  INT32_C(   554032757),  INT32_C(  1515037457) },
      { -INT32_C(   666779535),  INT32_C(   659722573), -INT32_C(  2140717693),  INT32_C(   919972059), -INT32_C(   799100951), -INT32_C(   431716239),  INT32_C(  1996966246),  INT32_C(  1590777069) },
      {  INT32_C(   806780940),  INT32_C(   446209606),  INT32_C(  1209892443), -INT32_C(  1472995414),  INT32_C(    12408833),  INT32_C(  1634974101), -INT32_C(  1280078845), -INT32_C(  1640780654) } },
    { {  INT32_C(  1630933780),  INT32_C(   663325092), -INT32_C(  1565986618), -INT32_C(  2099741287), -INT32_C(  1655491028),  INT32_C(   629381055), -INT32_C(   845313312), -INT32_C(   215191841) },
      UINT8_C(129),
      { -INT32_C(   349874846), -INT32_C(   827175458),  INT32_C(  1919374581),  INT32_C(  1671424556), -INT32_C(   735953859),  INT32_C(  1253328832),  INT32_C(  1395229412),  INT32_C(   282336686) },
      {  INT32_C(  1358756466),  INT32_C(  1008643655),  INT32_C(   783189506), -INT32_C(  1382986384),  INT32_C(  1250014090), -INT32_C(   543934981),  INT32_C(  1697824439),  INT32_C(  1299580891) },
      {  INT32_C(  1431878345),  INT32_C(   663325092), -INT32_C(  1565986618), -INT32_C(  2099741287), -INT32_C(  1655491028),  INT32_C(   629381055), -INT32_C(   845313312),  INT32_C(   444839362) } },
    { {  INT32_C(  1218277889),  INT32_C(   579124000), -INT32_C(  1320144319),  INT32_C(   173990528), -INT32_C(  1873485931), -INT32_C(   865015787), -INT32_C(  2127387994), -INT32_C(  1429296983) },
      UINT8_C( 26),
      {  INT32_C(   658174571), -INT32_C(  1452778378), -INT32_C(  1909908820),  INT32_C(  1478767480),  INT32_C(  1869460615), -INT32_C(   954844892),  INT32_C(   326145899), -INT32_C(   785573018) },
      { -INT32_C(  2064095475),  INT32_C(  1882022339), -INT32_C(   201435525),  INT32_C(   273359497), -INT32_C(    75515689),  INT32_C(  1573033458), -INT32_C(  1821297875),  INT32_C(  1533320782) },
      {  INT32_C(  1218277889), -INT32_C(   718468210), -INT32_C(  1320144319), -INT32_C(  1137962471), -INT32_C(   613871906), -INT32_C(   865015787), -INT32_C(  2127387994), -INT32_C(  1429296983) } },
    { { -INT32_C(   908108538),  INT32_C(   960040126), -INT32_C(   349423518),  INT32_C(   838563930),  INT32_C(   573406000),  INT32_C(  1031859984),  INT32_C(  1892741410), -INT32_C(  1781779057) },
      UINT8_C(146),
      { -INT32_C(  1202692181), -INT32_C(   803567208),  INT32_C(   757794485),  INT32_C(  2086558721),  INT32_C(  2039316617), -INT32_C(   241448448), -INT32_C(   796848997), -INT32_C(  2090723624) },
      {  INT32_C(   238793333), -INT32_C(   253864389),  INT32_C(  1562249564), -INT32_C(   270893978), -INT32_C(    43489284), -INT32_C(   856816591), -INT32_C(   392401136), -INT32_C(    76808570) },
      { -INT32_C(   908108538),  INT32_C(    55653146), -INT32_C(   349423518),  INT32_C(   838563930), -INT32_C(  1730672489),  INT32_C(  1031859984),  INT32_C(  1892741410),  INT32_C(  1645054043) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi32(test_vec[i].src);
    simde__mmask8 k = test_vec[i].k;
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_rorv_epi32(src, k, a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i src = simde_test_x86_random_i32x8();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m256i a = simde_test_x86_random_i32x8();
    simde__m256i b = simde_test_x86_random_i32x8();
    simde__m256i r = simde_mm256_mask_rorv_epi32(src, k, a, b);

    simde_test_x86_write_i32x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_maskz_rorv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int32_t a[8];
    const int32_t b[8];
    const int32_t r[8];
  } test_vec[] = {
    { UINT8_C(176),
      { -INT32_C(    34928217), -INT32_C(   245768984),  INT32_C(  1985459962), -INT32_C(   143505520), -INT32_C(  1289130065), -INT32_C(   842795682), -INT32_C(  1873564782), -INT32_C(  1103016169) },
      {  INT32_C(  1086008408),  INT32_C(    36836360),  INT32_C(  1534626250), -INT32_C(  2142049328), -INT32_C(  1204585638),  INT32_C(    59176817), -INT32_C(  1181492830), -INT32_C(  2122853336) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   899945492), -INT32_C(    89168159),  INT32_C(           0),  INT32_C(   398344527) } },
    { UINT8_C(  1),
      {  INT32_C(  1175044402),  INT32_C(  2098269427),  INT32_C(  1867345028),  INT32_C(   986369470), -INT32_C(   122977791), -INT32_C(   493179384),  INT32_C(   369775681), -INT32_C(    32011317) },
      {  INT32_C(  1078206541), -INT32_C(  1312991956), -INT32_C(  2145383487), -INT32_C(   642061608),  INT32_C(  1976657261),  INT32_C(  1415015443), -INT32_C(  1955962176),  INT32_C(   982090221) },
      {  INT32_C(   160575566),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT8_C(162),
      {  INT32_C(   600734670),  INT32_C(  1139113784), -INT32_C(  1977850464), -INT32_C(  2064124641),  INT32_C(   865562055),  INT32_C(   653519812), -INT32_C(   686522794), -INT32_C(   696693240) },
      {  INT32_C(    49891529),  INT32_C(  1732632263),  INT32_C(  1660051779),  INT32_C(   518449495),  INT32_C(   458325334), -INT32_C(  1086241687), -INT32_C(   896117310),  INT32_C(  1839206564) },
      {  INT32_C(           0),  INT32_C(  1887947518),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   502040075),  INT32_C(           0), -INT32_C(  1922591520) } },
    { UINT8_C( 88),
      {  INT32_C(  2015326106),  INT32_C(   381454261),  INT32_C(  1651318393), -INT32_C(  2101769467),  INT32_C(   552326364),  INT32_C(  1810082325),  INT32_C(  1359981889), -INT32_C(   391545778) },
      { -INT32_C(  1587492628), -INT32_C(   927523761),  INT32_C(  1076569403), -INT32_C(  1916607312), -INT32_C(   844255560), -INT32_C(  1724346280), -INT32_C(  1930803394), -INT32_C(  1334537276) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1962573127), -INT32_C(   338371552),  INT32_C(           0),  INT32_C(  1144960261),  INT32_C(           0) } },
    { UINT8_C( 92),
      { -INT32_C(   240430636),  INT32_C(   757888008), -INT32_C(  2082575201), -INT32_C(   600085714), -INT32_C(  1456142312), -INT32_C(  1998074303),  INT32_C(  1280078777), -INT32_C(  1146553113) },
      {  INT32_C(  1454134349),  INT32_C(  1736693960),  INT32_C(  1944740164), -INT32_C(   447797812),  INT32_C(  1854833709),  INT32_C(   217544019), -INT32_C(   816299032), -INT32_C(  1920335552) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(   130160951), -INT32_C(  1293040714),  INT32_C(  1086671272),  INT32_C(           0), -INT32_C(  1186182029),  INT32_C(           0) } },
    { UINT8_C( 85),
      {  INT32_C(   253616951), -INT32_C(   933985177), -INT32_C(  1818900626), -INT32_C(  1681884650),  INT32_C(  2112761608),  INT32_C(  1768290854),  INT32_C(  1403597906), -INT32_C(   173525058) },
      { -INT32_C(  2130328294), -INT32_C(  1219864247),  INT32_C(   910876448),  INT32_C(  1641089881),  INT32_C(  1625210682),  INT32_C(   214516665),  INT32_C(   895447927), -INT32_C(  1003812694) },
      { -INT32_C(   948384317),  INT32_C(           0), -INT32_C(  1818900626),  INT32_C(           0),  INT32_C(  2072756767),  INT32_C(           0),  INT32_C(  1382589607),  INT32_C(           0) } },
    { UINT8_C(205),
      { -INT32_C(  1994963408),  INT32_C(  1873399440),  INT32_C(   617144089),  INT32_C(  1885219249),  INT32_C(  1244315143), -INT32_C(    88000889), -INT32_C(  1650133355),  INT32_C(  1365993761) },
      {  INT32_C(  1071284911),  INT32_C(  1773044560),  INT32_C(   344815203), -INT32_C(  1501238369),  INT32_C(   837856937),  INT32_C(  2049683940), -INT32_C(   921186137), -INT32_C(   400915911) },
      { -INT32_C(  1939795410),  INT32_C(           0),  INT32_C(   614013923), -INT32_C(   524528798),  INT32_C(           0),  INT32_C(           0),  INT32_C(   725305837), -INT32_C(  1246457688) } },
    { UINT8_C(  4),
      {  INT32_C(  2018846709),  INT32_C(  1272692437),  INT32_C(   938143564),  INT32_C(   585207924),  INT32_C(   822547072),  INT32_C(   232292669),  INT32_C(   457613720), -INT32_C(  1323356484) },
      {  INT32_C(   707359829),  INT32_C(  2138375475),  INT32_C(  1756782580),  INT32_C(  1871419375), -INT32_C(   408907095), -INT32_C(  1410041581), -INT32_C(   691651814), -INT32_C(  1081612695) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(  1359690882),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__mmask8 k = test_vec[i].k;
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_rorv_epi32(k, a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m256i a = simde_test_x86_random_i32x8();
    simde__m256i b = simde_test_x86_random_i32x8();
    simde__m256i r = simde_mm256_maskz_rorv_epi32(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_rorv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(   905140135), -INT32_C(  1203783225), -INT32_C(   373183298), -INT32_C(  2086456397), -INT32_C(  2069725447),  INT32_C(  1028941064),  INT32_C(   144779901),  INT32_C(   426948722),
        -INT32_C(   682662384), -INT32_C(   577794529), -INT32_C(   288993221),  INT32_C(  2087807363), -INT32_C(   285207321), -INT32_C(   164866951), -INT32_C(   234959490), -INT32_C(  1794477947) },
      { -INT32_C(   177382954),  INT32_C(   601029864), -INT32_C(   804153267), -INT32_C(   397573631),  INT32_C(   265768342),  INT32_C(   537199522),  INT32_C(  1443955664),  INT32_C(  1256922228),
         INT32_C(  1564498037), -INT32_C(  1568665003), -INT32_C(  1401777749), -INT32_C(  1433026796), -INT32_C(  1346737139),  INT32_C(  1087356784),  INT32_C(   932635075),  INT32_C(  1921090301) },
      { -INT32_C(   849437481), -INT32_C(   944226375),  INT32_C(  1710706189), -INT32_C(  1043228199), -INT32_C(  1979980270),  INT32_C(   257235266),  INT32_C(   712837281),  INT32_C(   730276247),
         INT32_C(  2066777786),  INT32_C(  2087780076),  INT32_C(   125688010),  INT32_C(   379074503), -INT32_C(  1489012736),  INT32_C(  1417279020), -INT32_C(   566240849), -INT32_C(  1470921684) } },
    { {  INT32_C(   802210266),  INT32_C(  2144424148), -INT32_C(   164936478),  INT32_C(   278970371), -INT32_C(  1648403923), -INT32_C(   572682470),  INT32_C(  1830057072), -INT32_C(   790653450),
         INT32_C(   721465175), -INT32_C(   492121856),  INT32_C(   433640725), -INT32_C(  1020692330), -INT32_C(   312415789), -INT32_C(   389399176), -INT32_C(  1487544655), -INT32_C(   898091917) },
      { -INT32_C(   453674780),  INT32_C(  1606852425),  INT32_C(   192454261), -INT32_C(   372334314),  INT32_C(    47591050),  INT32_C(   501915755), -INT32_C(   222019714),  INT32_C(  1488731252),
        -INT32_C(    29576524), -INT32_C(   966983087), -INT32_C(  1227762272),  INT32_C(    27303798),  INT32_C(   956528333), -INT32_C(  1789465065), -INT32_C(  1584981459),  INT32_C(   200885335) },
      { -INT32_C(  1560474595),  INT32_C(  1782573224),  INT32_C(  1512511409), -INT32_C(  2097148862), -INT32_C(  1956155434), -INT32_C(   480527439), -INT32_C(  1269706303), -INT32_C(   111186675),
         INT32_C(   183857840),  INT32_C(  1753248085),  INT32_C(   433640725), -INT32_C(  1511892212),  INT32_C(  1319070471), -INT32_C(  1803882031), -INT32_C(   175293778), -INT32_C(   261560428) } },
    { {  INT32_C(  1191785974), -INT32_C(   686922185), -INT32_C(  1299259589),  INT32_C(  1286811263), -INT32_C(  1148864860), -INT32_C(   766387292),  INT32_C(  1299437814),  INT32_C(   307784988),
        -INT32_C(   631610718),  INT32_C(    78735560), -INT32_C(   961134777),  INT32_C(   303262061), -INT32_C(   993159137),  INT32_C(  1788223092),  INT32_C(   330762743),  INT32_C(   421924726) },
      {  INT32_C(   989036657),  INT32_C(   792634600),  INT32_C(  1375139044),  INT32_C(  2086865245),  INT32_C(   356528545),  INT32_C(  1182783055),  INT32_C(  1448752864), -INT32_C(  1217429434),
        -INT32_C(   386833920), -INT32_C(   350802169), -INT32_C(  2143548125), -INT32_C(  1191337962),  INT32_C(   550321617), -INT32_C(   194556652), -INT32_C(   917847677),  INT32_C(  1099020865) },
      { -INT32_C(  1694817404),  INT32_C(   936840806), -INT32_C(  1154945549),  INT32_C(  1704555514),  INT32_C(  1573051218), -INT32_C(  1219910493),  INT32_C(  1299437814),  INT32_C(  1883857332),
        -INT32_C(   631610718), -INT32_C(  1878433071), -INT32_C(   120141848),  INT32_C(  1302705224), -INT32_C(   871374234),  INT32_C(  1642546857), -INT32_C(   495525570),  INT32_C(   210962363) } },
    { {  INT32_C(   589918748), -INT32_C(   988921694),  INT32_C(  1682328398), -INT32_C(  1138998293), -INT32_C(  1797461376), -INT32_C(  1165475017),  INT32_C(  1166332676), -INT32_C(  1450834547),
         INT32_C(   432910199),  INT32_C(  1054858224),  INT32_C(   295839014), -INT32_C(   389169304), -INT32_C(   528635223), -INT32_C(   224786963),  INT32_C(  1698111192), -INT32_C(  1693467101) },
      {  INT32_C(  1572133997), -INT32_C(   560229449),  INT32_C(   552550072), -INT32_C(  1509310979),  INT32_C(  1418102375),  INT32_C(  1682317451),  INT32_C(  1657372223), -INT32_C(  1442981828),
         INT32_C(  1812443828), -INT32_C(    28663227), -INT32_C(   585221664),  INT32_C(  1568876534),  INT32_C(   967969197),  INT32_C(  1771960362), -INT32_C(  1295292810), -INT32_C(   212022977) },
      { -INT32_C(  1864296117),  INT32_C(   478233994),  INT32_C(  1179340388), -INT32_C(   522051747),  INT32_C(    19511765),  INT32_C(  1727484168), -INT32_C(  1962301944), -INT32_C(  1738516262),
        -INT32_C(   621317732), -INT32_C(  2114519329),  INT32_C(   295839014),  INT32_C(   922592163),  INT32_C(  1431241709),  INT32_C(  2071766657), -INT32_C(   595893868),  INT32_C(   908033095) } },
    { { -INT32_C(  1067490437), -INT32_C(   423712507), -INT32_C(   641475101), -INT32_C(  1305000188),  INT32_C(  2062281040),  INT32_C(  1491306721),  INT32_C(   755675118), -INT32_C(   215914888),
        -INT32_C(   810319671),  INT32_C(   229995050),  INT32_C(  1407613007),  INT32_C(   268770751), -INT32_C(   393547770),  INT32_C(  1732275832), -INT32_C(  1785443811),  INT32_C(  2055779761) },
      {  INT32_C(  1615412022), -INT32_C(    59900243),  INT32_C(   911233911),  INT32_C(  2001098097), -INT32_C(  1101016762),  INT32_C(  1545969471), -INT32_C(  1678656790), -INT32_C(  1525319313),
         INT32_C(  1644519348), -INT32_C(   732007843),  INT32_C(   923512518),  INT32_C(  1252937988),  INT32_C(  1627917858), -INT32_C(  1749209427),  INT32_C(  1462939368), -INT32_C(   587380697) },
      {  INT32_C(  2106453761),  INT32_C(  1211053557), -INT32_C(  2017736781), -INT32_C(  1551738597),  INT32_C(  1105964965), -INT32_C(  1312353854), -INT32_C(    74759509), -INT32_C(   856562110),
         INT32_C(   940350715),  INT32_C(  1839960400),  INT32_C(  1028626913), -INT32_C(   251637285), -INT32_C(  1172128767),  INT32_C(  1942174211),  INT32_C(   496342090),  INT32_C(  1660227947) } },
    { {  INT32_C(    71172775),  INT32_C(  1004117109),  INT32_C(  1332929611),  INT32_C(  1486430261), -INT32_C(   591814354), -INT32_C(  1217169457),  INT32_C(  1292871205), -INT32_C(  1792471826),
        -INT32_C(  2087098610),  INT32_C(  1321103875), -INT32_C(  1935855530),  INT32_C(  2145662545), -INT32_C(  1487168040),  INT32_C(   979357460),  INT32_C(  1669820021), -INT32_C(  1996967814) },
      {  INT32_C(   437031191),  INT32_C(  1516882436),  INT32_C(  1290143483),  INT32_C(   365677117),  INT32_C(  2092705640),  INT32_C(  1807096822),  INT32_C(    63847817),  INT32_C(    76334829),
         INT32_C(  1528797271),  INT32_C(  1572243554), -INT32_C(   878076786), -INT32_C(   807373465), -INT32_C(  1840538212),  INT32_C(  1123877560),  INT32_C(   759548735), -INT32_C(   399388271) },
      {  INT32_C(  2080722440),  INT32_C(  1404934599), -INT32_C(   295925399), -INT32_C(   993459798),  INT32_C(   786217377), -INT32_C(   841007395),  INT32_C(   312903635),  INT32_C(  1735698760),
         INT32_C(   852368647), -INT32_C(   743465856), -INT32_C(  1051053452), -INT32_C(  1560295316),  INT32_C(  1975115146),  INT32_C(  1607406650), -INT32_C(   955327254),  INT32_C(  1480410236) } },
    { { -INT32_C(   867938199),  INT32_C(  1730804440), -INT32_C(    47000682), -INT32_C(   456387768),  INT32_C(  1752570032),  INT32_C(  1504342810), -INT32_C(   813240258),  INT32_C(   733460673),
        -INT32_C(   503842040), -INT32_C(  1941430027),  INT32_C(   998865651),  INT32_C(  1025463949), -INT32_C(  2002348690),  INT32_C(  1189171208),  INT32_C(    34957376),  INT32_C(   674090272) },
      { -INT32_C(  1106697016),  INT32_C(   944394564),  INT32_C(  1500763084), -INT32_C(  1751674071),  INT32_C(   824130856), -INT32_C(   831061619), -INT32_C(  1982821015),  INT32_C(   582090074),
         INT32_C(  1709226785), -INT32_C(   660788724),  INT32_C(   657592830), -INT32_C(   859846236), -INT32_C(  1812079098),  INT32_C(  1214346719),  INT32_C(  1557213442),  INT32_C(  1333691182) },
      {  INT32_C(  1774994512), -INT32_C(  2039308371),  INT32_C(   963629869), -INT32_C(  1536006647), -INT32_C(  1335331304), -INT32_C(  1731015341),  INT32_C(   526893944), -INT32_C(   303157174),
         INT32_C(  1895562628),  INT32_C(   257475714), -INT32_C(   299504692), -INT32_C(   741214872), -INT32_C(  1172137387), -INT32_C(  1916624880),  INT32_C(     8739344),  INT32_C(   880844983) } },
    { {  INT32_C(  1253400382), -INT32_C(  2027728247),  INT32_C(   128931171),  INT32_C(   617901598),  INT32_C(   750244172),  INT32_C(  1215568198),  INT32_C(  2040809034),  INT32_C(   130556873),
         INT32_C(   189889922),  INT32_C(   865301712), -INT32_C(   415546679), -INT32_C(    66383952),  INT32_C(   656983008),  INT32_C(   644849116), -INT32_C(  1398860573), -INT32_C(  1179424713) },
      { -INT32_C(  1245444891),  INT32_C(  1122588537),  INT32_C(  1227498649),  INT32_C(   323368243), -INT32_C(   734368008), -INT32_C(   285496821), -INT32_C(   174417218), -INT32_C(   391230206),
        -INT32_C(   878874030),  INT32_C(  1661830858), -INT32_C(   575850582),  INT32_C(  1710355309),  INT32_C(  1832528737), -INT32_C(  1822738987), -INT32_C(   762775857), -INT32_C(  1782958525) },
      { -INT32_C(   229266695), -INT32_C(  1851177789), -INT32_C(   676679293), -INT32_C(  1916549990), -INT32_C(  1211020244),  INT32_C(   684265091), -INT32_C(   426698455),  INT32_C(  1106381042),
         INT32_C(  1600160468),  INT32_C(   873260253), -INT32_C(  1300640048),  INT32_C(  2106056792),  INT32_C(   328491504),  INT32_C(  2096030003),  INT32_C(   700930366), -INT32_C(   147428090) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_rorv_epi32(a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m512i b = simde_test_x86_random_i32x16();
    simde__m512i r = simde_mm512_rorv_epi32(a, b);

    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_rorv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(  1184851916), -INT32_C(  1348346192),  INT32_C(   338407473),  INT32_C(   504908348),  INT32_C(   541493486),  INT32_C(  1262965618), -INT32_C(   469153697), -INT32_C(  1754933301),
        -INT32_C(   287439554),  INT32_C(   345931747),  INT32_C(  1797835055),  INT32_C(    25837587),  INT32_C(   992071881), -INT32_C(  2105120733),  INT32_C(  2087096240), -INT32_C(  1542206362) },
      UINT16_C(61905),
      {  INT32_C(   829469843), -INT32_C(   235233080), -INT32_C(  1791947253),  INT32_C(   828832271), -INT32_C(  1130788555),  INT32_C(  1917536780), -INT32_C(   650202426), -INT32_C(   372568234),
        -INT32_C(  1944437821), -INT32_C(   411232805),  INT32_C(   847031843), -INT32_C(   563879255),  INT32_C(  2023423340),  INT32_C(   233498183), -INT32_C(   286905960), -INT32_C(    52907720) },
      { -INT32_C(   930549011),  INT32_C(   716113159),  INT32_C(  1549610163),  INT32_C(  2050736142),  INT32_C(    83023293),  INT32_C(  1393679803),  INT32_C(  1044576262), -INT32_C(  1774576983),
         INT32_C(   341819916),  INT32_C(  2067664583),  INT32_C(  1238866746),  INT32_C(   432214620), -INT32_C(  1558268184), -INT32_C(  1711853677), -INT32_C(   774424281),  INT32_C(  1600590419) },
      { -INT32_C(  1533441147), -INT32_C(  1348346192),  INT32_C(   338407473),  INT32_C(   504908348), -INT32_C(   456373843),  INT32_C(  1262965618),  INT32_C(   459602635), -INT32_C(  1418402425),
        -INT32_C(  1137131101),  INT32_C(   345931747),  INT32_C(  1797835055),  INT32_C(    25837587),  INT32_C(  1819843325),  INT32_C(  1556668861),  INT32_C(   836619347),  INT32_C(   371662747) } },
    { { -INT32_C(  1670134060),  INT32_C(   253211349), -INT32_C(  1453789619), -INT32_C(   373089279),  INT32_C(  1703731666),  INT32_C(   956203792),  INT32_C(   252302780), -INT32_C(  1133547289),
         INT32_C(   207151671), -INT32_C(   501518444),  INT32_C(  1586197597),  INT32_C(  1648840336),  INT32_C(  1086837807),  INT32_C(   343459159), -INT32_C(  2111602278),  INT32_C(   675189489) },
      UINT16_C(38517),
      {  INT32_C(  1342507316),  INT32_C(  2009359083),  INT32_C(   147150017),  INT32_C(  2111632822), -INT32_C(  1388170187),  INT32_C(  1798233160), -INT32_C(  1661067426),  INT32_C(  2083681096),
         INT32_C(  1758214012),  INT32_C(  1541378202), -INT32_C(  1704745756), -INT32_C(   837336935), -INT32_C(  1132766860), -INT32_C(  1809340106),  INT32_C(   288433609),  INT32_C(   361587608) },
      {  INT32_C(   897407643), -INT32_C(   812622614), -INT32_C(  1704332544), -INT32_C(  1486323405),  INT32_C(   274982106),  INT32_C(  1453689741),  INT32_C(  1231541936), -INT32_C(   731974343),
         INT32_C(   956947279),  INT32_C(   923310391), -INT32_C(  1060015475), -INT32_C(   848807181), -INT32_C(  1411462370),  INT32_C(   117605206), -INT32_C(  1840223911), -INT32_C(  1385714082) },
      {  INT32_C(    10561162),  INT32_C(   253211349),  INT32_C(   147150017), -INT32_C(   373089279),  INT32_C(  1351421291), -INT32_C(   498902666),  INT32_C(   526294270), -INT32_C(  1133547289),
         INT32_C(   207151671), -INT32_C(  1088342857),  INT32_C(   656724765),  INT32_C(  1648840336), -INT32_C(   236100142),  INT32_C(   343459159), -INT32_C(  2111602278),  INT32_C(  1446350432) } },
    { { -INT32_C(  1058574199), -INT32_C(  1745359094),  INT32_C(  1415039329),  INT32_C(   555925251), -INT32_C(   506724213), -INT32_C(   588722557), -INT32_C(  1771161544),  INT32_C(  1866782182),
         INT32_C(  1345268550),  INT32_C(  2095523610), -INT32_C(   204456208), -INT32_C(  1995115778),  INT32_C(  1969938674), -INT32_C(   414100561),  INT32_C(  1920843915), -INT32_C(   605961835) },
      UINT16_C( 4588),
      {  INT32_C(   305661739),  INT32_C(  1397827715),  INT32_C(   793136923),  INT32_C(  1108293848), -INT32_C(     6963538),  INT32_C(   599728549),  INT32_C(  1961121171),  INT32_C(  1552273712),
         INT32_C(  1533984216),  INT32_C(    11452389), -INT32_C(   433064946), -INT32_C(   618053843), -INT32_C(  1545945347),  INT32_C(  1925618399),  INT32_C(   535276271),  INT32_C(  1400597627) },
      {  INT32_C(   263121450), -INT32_C(  1206886999),  INT32_C(  2124300113),  INT32_C(  2086258558),  INT32_C(  1696543878), -INT32_C(  1109858866),  INT32_C(   199016335),  INT32_C(  1432246315),
        -INT32_C(   345699006), -INT32_C(  1146915478),  INT32_C(   859456180), -INT32_C(  1884318967), -INT32_C(  1779052857),  INT32_C(  1112722867), -INT32_C(  1219678324), -INT32_C(   921850745) },
      { -INT32_C(  1058574199), -INT32_C(  1745359094),  INT32_C(   663590819),  INT32_C(   138208097), -INT32_C(   506724213), -INT32_C(  2037084420), -INT32_C(  1423513144),  INT32_C(   638292154),
         INT32_C(   383496054),  INT32_C(  2095523610), -INT32_C(   204456208), -INT32_C(  1995115778), -INT32_C(    79186563), -INT32_C(   414100561),  INT32_C(  1920843915), -INT32_C(   605961835) } },
    { {  INT32_C(   599028409), -INT32_C(  1679927065), -INT32_C(  1546774374),  INT32_C(  1932754348), -INT32_C(    32954293), -INT32_C(  2126423051),  INT32_C(   288919178), -INT32_C(   203799238),
        -INT32_C(  1642688585), -INT32_C(  2126842393), -INT32_C(  1188820979), -INT32_C(   785557626),  INT32_C(  1959802495),  INT32_C(   485822865), -INT32_C(   634573153),  INT32_C(   734857331) },
      UINT16_C(58519),
      {  INT32_C(    64585417),  INT32_C(   588048127), -INT32_C(   847539552),  INT32_C(   855898723),  INT32_C(  1682216047), -INT32_C(   560798544), -INT32_C(  1947859266), -INT32_C(    93356751),
        -INT32_C(    83998469), -INT32_C(   820180433), -INT32_C(     6514532),  INT32_C(    36937619), -INT32_C(   463046861),  INT32_C(   432208219),  INT32_C(   816163071),  INT32_C(   539628581) },
      { -INT32_C(  1944442787), -INT32_C(   832882894),  INT32_C(  1674508240), -INT32_C(   916127338), -INT32_C(   726807688), -INT32_C(  1007849532),  INT32_C(  1022595607),  INT32_C(    73146023),
         INT32_C(  2039510598),  INT32_C(  2118642606),  INT32_C(  2028017122), -INT32_C(  1891547626), -INT32_C(   698093807),  INT32_C(  1989759070), -INT32_C(  1968009757), -INT32_C(   242348117) },
      {  INT32_C(   516683336),  INT32_C(   968870083), -INT32_C(  1834955397),  INT32_C(  1932754348),  INT32_C(  1150578532), -INT32_C(  2126423051),  INT32_C(   288919178),  INT32_C(  1676992250),
        -INT32_C(  1642688585), -INT32_C(  2126842393),  INT32_C(  1072113191), -INT32_C(   785557626),  INT32_C(  1959802495),  INT32_C(  1728832876), -INT32_C(   434850529), -INT32_C(  2069625534) } },
    { {  INT32_C(   862592645), -INT32_C(   340676087), -INT32_C(   563899705), -INT32_C(   361913128), -INT32_C(   222244717),  INT32_C(    57170208), -INT32_C(  1853023514), -INT32_C(  1350362327),
         INT32_C(  1122168121),  INT32_C(  1731040415), -INT32_C(    12218330), -INT32_C(   924208587), -INT32_C(  1564825214), -INT32_C(   374988286),  INT32_C(  1719284541), -INT32_C(  2011824817) },
      UINT16_C(63723),
      { -INT32_C(   125007157),  INT32_C(   914993905), -INT32_C(  1696022863),  INT32_C(  1094937222), -INT32_C(  1285339635), -INT32_C(  1444503505),  INT32_C(   480720135), -INT32_C(  1995140418),
         INT32_C(   243376156),  INT32_C(    71567955),  INT32_C(  1335832008), -INT32_C(  1534008681),  INT32_C(  1465381672), -INT32_C(  1694417260),  INT32_C(   834119795),  INT32_C(  1455082554) },
      { -INT32_C(  1083949972),  INT32_C(   264546630),  INT32_C(  1834902486),  INT32_C(  1846734405),  INT32_C(  1975871969),  INT32_C(   454084263), -INT32_C(  1471362962),  INT32_C(    16713620),
        -INT32_C(  1983880381), -INT32_C(   493321204),  INT32_C(   743503591), -INT32_C(   962960668),  INT32_C(  1933271243), -INT32_C(  1785836506), -INT32_C(  1472341484),  INT32_C(   615005409) },
      { -INT32_C(  1396733752), -INT32_C(   992336181), -INT32_C(   563899705),  INT32_C(   839523156), -INT32_C(   222244717),  INT32_C(  1599327552), -INT32_C(   736333156),  INT32_C(  1227614353),
         INT32_C(  1122168121),  INT32_C(  1731040415), -INT32_C(    12218330),  INT32_C(  2051608105),  INT32_C(  1695214334),  INT32_C(  1382810874),  INT32_C(  2055680795),  INT32_C(   727541277) } },
    { { -INT32_C(  1397856096), -INT32_C(   745584916),  INT32_C(   553705277),  INT32_C(   216504897),  INT32_C(   561980410), -INT32_C(  2085221009), -INT32_C(   919866392), -INT32_C(   789654736),
         INT32_C(   679255100),  INT32_C(   536611810),  INT32_C(   725744874), -INT32_C(  1858590570), -INT32_C(  1162692789), -INT32_C(  1388484411), -INT32_C(  1955174309),  INT32_C(  2019255356) },
      UINT16_C(55296),
      { -INT32_C(  1662786656),  INT32_C(  1134087682),  INT32_C(   829173753),  INT32_C(  1927919552), -INT32_C(  1344622990), -INT32_C(   786942629),  INT32_C(   473322689),  INT32_C(  1827944140),
         INT32_C(   470407193), -INT32_C(  1621122650), -INT32_C(  1865298992), -INT32_C(   184370557), -INT32_C(  1012605848), -INT32_C(   745227246), -INT32_C(   588264944),  INT32_C(   441050113) },
      {  INT32_C(  1647727292), -INT32_C(  1006528781), -INT32_C(   464203167), -INT32_C(   187082868),  INT32_C(  1169653299),  INT32_C(  1259883578),  INT32_C(   388433942), -INT32_C(  1473089300),
        -INT32_C(  1240831806),  INT32_C(  1618611198),  INT32_C(  1782894302),  INT32_C(  1499405861), -INT32_C(   694282596),  INT32_C(  2032252770), -INT32_C(  1416607297),  INT32_C(  2085864121) },
      { -INT32_C(  1397856096), -INT32_C(   745584916),  INT32_C(   553705277),  INT32_C(   216504897),  INT32_C(   561980410), -INT32_C(  2085221009), -INT32_C(   919866392), -INT32_C(   789654736),
         INT32_C(   679255100),  INT32_C(   536611810),  INT32_C(   725744874),  INT32_C(   531109332),  INT32_C(   978175628), -INT32_C(  1388484411), -INT32_C(  1176529887),  INT32_C(   619839629) } },
    { {  INT32_C(   691166507),  INT32_C(  1183427688), -INT32_C(  1598960006), -INT32_C(  1996943124), -INT32_C(  1990289626),  INT32_C(   218267726), -INT32_C(  2101833015), -INT32_C(  2130834603),
        -INT32_C(   777441176),  INT32_C(  1461138396), -INT32_C(   302528511), -INT32_C(     9047848), -INT32_C(   695676025),  INT32_C(   484674131),  INT32_C(  1906285596),  INT32_C(   284269991) },
      UINT16_C(39886),
      { -INT32_C(   120673567), -INT32_C(   121516287),  INT32_C(   837327292),  INT32_C(   537161880),  INT32_C(   699029574),  INT32_C(   331728756),  INT32_C(   716270904),  INT32_C(  1590001533),
         INT32_C(   727094057),  INT32_C(   505616226),  INT32_C(  1229917361), -INT32_C(  1033219204),  INT32_C(   552277163),  INT32_C(   338932187), -INT32_C(  1690377186), -INT32_C(  1913060509) },
      { -INT32_C(   122138730),  INT32_C(   404151143),  INT32_C(  1667327463),  INT32_C(  1680198584), -INT32_C(  1148972576), -INT32_C(   523257918), -INT32_C(     8712804), -INT32_C(  1500744688),
         INT32_C(   715015363),  INT32_C(   121812000), -INT32_C(   764697575),  INT32_C(  1328975983),  INT32_C(  1661713057),  INT32_C(   222550641), -INT32_C(   133316888), -INT32_C(   157312718) },
      {  INT32_C(   691166507),  INT32_C(    66159518),  INT32_C(  2019807539),  INT32_C(    74487840), -INT32_C(  1990289626),  INT32_C(   218267726), -INT32_C(  1424567422),  INT32_C(  2138922693),
         INT32_C(   627757669),  INT32_C(   505616226), -INT32_C(   302528511), -INT32_C(  1493596972), -INT32_C(  1871345067),  INT32_C(   484674131),  INT32_C(  1906285596),  INT32_C(  1087955838) } },
    { { -INT32_C(    31441442),  INT32_C(   184902642),  INT32_C(  1994223623), -INT32_C(  1564077311),  INT32_C(  1057411533), -INT32_C(  1823716949),  INT32_C(   982276360), -INT32_C(   785372173),
         INT32_C(  1523536232), -INT32_C(  1150888780),  INT32_C(  1177633605),  INT32_C(   619247447),  INT32_C(  1952706248),  INT32_C(  1074245688), -INT32_C(    59075831),  INT32_C(   667790270) },
      UINT16_C(40188),
      { -INT32_C(   411979647), -INT32_C(  1674856853), -INT32_C(   460029188),  INT32_C(   181623974), -INT32_C(   658895920), -INT32_C(   982793397), -INT32_C(  1922028864), -INT32_C(   768971695),
        -INT32_C(  2017813732),  INT32_C(  1294263633),  INT32_C(   221362279), -INT32_C(   451476460),  INT32_C(  1522389263),  INT32_C(  1428170901), -INT32_C(  1545433006),  INT32_C(   427166973) },
      { -INT32_C(   123653977),  INT32_C(  2084947221), -INT32_C(  1853261955), -INT32_C(  1955159685),  INT32_C(   132461426), -INT32_C(  1369701028), -INT32_C(  1823326314), -INT32_C(   206780341),
         INT32_C(   233524728), -INT32_C(  1886834414),  INT32_C(   589370024),  INT32_C(   648976307),  INT32_C(   657298635),  INT32_C(   802523801),  INT32_C(   348268489), -INT32_C(   418942993) },
      { -INT32_C(    31441442),  INT32_C(   184902642),  INT32_C(   614733799),  INT32_C(  1516999873), -INT32_C(  2097924562),  INT32_C(  1455174844), -INT32_C(  1062534603), -INT32_C(  1975892659),
         INT32_C(  1523536232), -INT32_C(  1150888780),  INT32_C(  1728917944), -INT32_C(   528311134),  INT32_C(   569071546),  INT32_C(  1074245688), -INT32_C(    59075831),  INT32_C(   435827436) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_rorv_epi32(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m512i b = simde_test_x86_random_i32x16();
    simde__m512i r = simde_mm512_mask_rorv_epi32(src, k, a, b);

    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_rorv_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(62396),
      {  INT32_C(  2099564532),  INT32_C(  2140195934), -INT32_C(  1642642449), -INT32_C(  1758272919), -INT32_C(   568210679), -INT32_C(  1123620101),  INT32_C(   103609854), -INT32_C(   772150819),
         INT32_C(   374283704), -INT32_C(   661200919), -INT32_C(  1938379486), -INT32_C(  1742493553),  INT32_C(  1870021748),  INT32_C(   741112878),  INT32_C(  1328699762), -INT32_C(    98555070) },
      {  INT32_C(   823160648),  INT32_C(  1896523342), -INT32_C(   486703021), -INT32_C(  1652940759), -INT32_C(  1844645532), -INT32_C(   541181843), -INT32_C(   751898223),  INT32_C(  1707953948),
         INT32_C(   227991230), -INT32_C(   662790012),  INT32_C(  1253735201), -INT32_C(     1624677), -INT32_C(  1819151578), -INT32_C(  1116516565),  INT32_C(  1569759552), -INT32_C(  1346216208) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(   394398782),  INT32_C(   885758321), -INT32_C(  1646125904),  INT32_C(  1474160695),  INT32_C(           0),  INT32_C(   530488797),
         INT32_C(  1497134816), -INT32_C(  1651937794),  INT32_C(           0),  INT32_C(           0), -INT32_C(   776087279), -INT32_C(  2050652785),  INT32_C(  1328699762),  INT32_C(   725809696) } },
    { UINT16_C(22588),
      {  INT32_C(   989446332),  INT32_C(  1404377752),  INT32_C(  1267224676), -INT32_C(   532763057), -INT32_C(  1271961279), -INT32_C(  1219071962), -INT32_C(  1894431027), -INT32_C(  1310240267),
        -INT32_C(  1427382255),  INT32_C(  1593680122),  INT32_C(  1068074480),  INT32_C(  1948247859),  INT32_C(  1999195985), -INT32_C(  1943109697), -INT32_C(  1155841083), -INT32_C(  1502870892) },
      { -INT32_C(   581936925), -INT32_C(   381923848),  INT32_C(    86566354),  INT32_C(   511264973),  INT32_C(  1452647063), -INT32_C(   404569055), -INT32_C(  1667039993), -INT32_C(   498987265),
         INT32_C(  1606455910), -INT32_C(  1320616737), -INT32_C(  1363775263),  INT32_C(  1338781624), -INT32_C(   207265070),  INT32_C(   769296166), -INT32_C(  2067170172), -INT32_C(   244970613) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(   337187554),  INT32_C(  1920926197),  INT32_C(  1590854504),  INT32_C(  1537947667),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   535245684),  INT32_C(  1406426570),  INT32_C(           0),  INT32_C(  1538372668),  INT32_C(           0) } },
    { UINT16_C( 9885),
      { -INT32_C(  1742570416), -INT32_C(   485948371),  INT32_C(  2132001202), -INT32_C(  1226709743), -INT32_C(  1287780392), -INT32_C(    30424524), -INT32_C(  1391871418), -INT32_C(    53237844),
         INT32_C(  1368782371), -INT32_C(  1405837574),  INT32_C(  1881884511),  INT32_C(    86445100),  INT32_C(  1220044051),  INT32_C(  1850140455),  INT32_C(  1310412706),  INT32_C(   441118454) },
      { -INT32_C(   563355676), -INT32_C(   578052227),  INT32_C(   323860199), -INT32_C(   703040317),  INT32_C(    18796761),  INT32_C(  1517249720), -INT32_C(  1431795020),  INT32_C(  1573188216),
         INT32_C(  1329278930), -INT32_C(  1238579505),  INT32_C(  1086945917), -INT32_C(   938024210),  INT32_C(  1791571378),  INT32_C(  1304705177),  INT32_C(   989359298),  INT32_C(   832027743) },
      {  INT32_C(   159524805),  INT32_C(           0),  INT32_C(  1694377859),  INT32_C(   920403106), -INT32_C(  1627132839),  INT32_C(           0),  INT32_C(           0), -INT32_C(   743985924),
         INT32_C(           0),  INT32_C(  1039489129), -INT32_C(  2124793093),  INT32_C(           0),  INT32_C(           0),  INT32_C(   594777015),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(54252),
      { -INT32_C(  1382433920),  INT32_C(   975640177),  INT32_C(  1830557014), -INT32_C(  1499279395), -INT32_C(    18990279), -INT32_C(  2123718519),  INT32_C(  1916651995),  INT32_C(  2068130298),
         INT32_C(  1428742116),  INT32_C(  1284526069),  INT32_C(  1119464548), -INT32_C(  1243063429),  INT32_C(   531875734),  INT32_C(  1117789543), -INT32_C(   508174617), -INT32_C(   329451001) },
      { -INT32_C(   817789735),  INT32_C(   941347283), -INT32_C(   109390722), -INT32_C(   978427345), -INT32_C(  1847238359),  INT32_C(  1725138303),  INT32_C(  1799850083),  INT32_C(  1549247619),
        -INT32_C(    64251864), -INT32_C(   399227286),  INT32_C(  1239526938),  INT32_C(   974098448),  INT32_C(  1892414705), -INT32_C(   573137287), -INT32_C(  1438114265), -INT32_C(   368664638) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(  1267706535), -INT32_C(  1615114939),  INT32_C(           0),  INT32_C(    47530259),  INT32_C(  1850194235),  INT32_C(  1332258111),
        -INT32_C(   464181025), -INT32_C(    44882925),  INT32_C(           0),  INT32_C(           0), -INT32_C(   473231399),  INT32_C(           0), -INT32_C(   809276483), -INT32_C(    82362751) } },
    { UINT16_C(12855),
      {  INT32_C(   444113382),  INT32_C(  1808307081), -INT32_C(   335816228), -INT32_C(   555750381),  INT32_C(   863787357),  INT32_C(  2119279414),  INT32_C(  1411191629), -INT32_C(   460958466),
         INT32_C(  2147483382),  INT32_C(  1860880273), -INT32_C(  1285954144),  INT32_C(   798046674), -INT32_C(   933097838), -INT32_C(    12143439), -INT32_C(   984390713), -INT32_C(  1381312073) },
      {  INT32_C(  1764600279),  INT32_C(   299308912), -INT32_C(   809225987), -INT32_C(    67217815),  INT32_C(   365125732), -INT32_C(   602666732),  INT32_C(   614557548),  INT32_C(   416369472),
         INT32_C(  1703018228),  INT32_C(   326522902), -INT32_C(   220054904), -INT32_C(   185737072),  INT32_C(  1426764096),  INT32_C(   657530554), -INT32_C(   968109434),  INT32_C(   316546078) },
      { -INT32_C(   247215052), -INT32_C(  1819710520),  INT32_C(  1608437479),  INT32_C(           0), -INT32_C(   751319659),  INT32_C(   439576549),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0), -INT32_C(  1424079429),  INT32_C(           0),  INT32_C(           0), -INT32_C(   933097838), -INT32_C(   777180033),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(24347),
      { -INT32_C(   306761353),  INT32_C(   640172100),  INT32_C(   520534066), -INT32_C(  1211087187), -INT32_C(   858420325), -INT32_C(    39887950),  INT32_C(     1686818),  INT32_C(  1197487311),
        -INT32_C(  1456203931), -INT32_C(  1982899113), -INT32_C(  1029057259), -INT32_C(  1216775908), -INT32_C(  1232843260), -INT32_C(   860675158), -INT32_C(  1345532448),  INT32_C(  1744186369) },
      { -INT32_C(  1693439164), -INT32_C(  1675239289), -INT32_C(   782315851),  INT32_C(  1284101959), -INT32_C(   805171930),  INT32_C(   278705712), -INT32_C(  2067830653), -INT32_C(   638863723),
         INT32_C(  1752497377), -INT32_C(  1861903908), -INT32_C(  1352440985),  INT32_C(  1643899963),  INT32_C(   691142137),  INT32_C(   909757875), -INT32_C(   893650891), -INT32_C(  1885100370) },
      {  INT32_C(  2128311063), -INT32_C(  2008264576),  INT32_C(           0),  INT32_C(  1534042253),  INT32_C(  1865635374),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
        -INT32_C(   728101966), -INT32_C(  1661614728),  INT32_C(   730157994), -INT32_C(   282123370),  INT32_C(  1109852763),  INT32_C(           0),  INT32_C(  1718551934),  INT32_C(           0) } },
    { UINT16_C( 6306),
      { -INT32_C(    38699272),  INT32_C(  1935677712), -INT32_C(  1017144376),  INT32_C(   767580412), -INT32_C(  1157991295),  INT32_C(  1706176682), -INT32_C(  1643355654), -INT32_C(   390680848),
         INT32_C(  1038444333),  INT32_C(  1219511936), -INT32_C(   569700383), -INT32_C(   401879961), -INT32_C(   358480320),  INT32_C(   810570806), -INT32_C(  1496425291),  INT32_C(   932086794) },
      {  INT32_C(  1819571436), -INT32_C(  1682627398), -INT32_C(  1703296973), -INT32_C(   863861364), -INT32_C(  1044962165),  INT32_C(   787548025),  INT32_C(  1842659427),  INT32_C(   816145220),
        -INT32_C(  1851975209),  INT32_C(  1898730046), -INT32_C(  1626561006), -INT32_C(  1234465237),  INT32_C(   746005171), -INT32_C(  1940232151),  INT32_C(  1845112616),  INT32_C(  1771937682) },
      {  INT32_C(           0), -INT32_C(   670677988),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   652716750),  INT32_C(           0),  INT32_C(   244017903),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1929576071),  INT32_C(  1086856532),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(15032),
      {  INT32_C(   646772474),  INT32_C(  1959567208), -INT32_C(  1459423426),  INT32_C(   617395629),  INT32_C(  1015936481),  INT32_C(  2087433602), -INT32_C(  1071907550),  INT32_C(  1627050854),
         INT32_C(   847677386),  INT32_C(  1688621862), -INT32_C(   133322678),  INT32_C(  1058855262),  INT32_C(  1367058895), -INT32_C(  2117212577),  INT32_C(  1229056227), -INT32_C(  2052506437) },
      { -INT32_C(   373870397), -INT32_C(   850567805),  INT32_C(  1690655238),  INT32_C(    61071924), -INT32_C(   363585653), -INT32_C(   378789627), -INT32_C(  1003311863), -INT32_C(  1404379927),
        -INT32_C(  1886060276),  INT32_C(  1700586079),  INT32_C(  1909006909), -INT32_C(  1888195580), -INT32_C(  1854224245), -INT32_C(   226826519),  INT32_C(  2109123731), -INT32_C(  1809252216) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   883240372), -INT32_C(  1138257474),  INT32_C(   333667756),  INT32_C(           0), -INT32_C(  1288667799),
         INT32_C(           0), -INT32_C(   917723572),  INT32_C(           0), -INT32_C(   470692459),  INT32_C(   971648885),  INT32_C(   801171187),  INT32_C(           0),  INT32_C(           0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_rorv_epi32(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m512i b = simde_test_x86_random_i32x16();
    simde__m512i r = simde_mm512_maskz_rorv_epi32(k, a, b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_rorv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[2];
    const int64_t b[2];
    const int64_t r[2];
  } test_vec[] = {
    { {  INT64_C( 7152693782144456081),  INT64_C( 2020249513474868575) },
      { -INT64_C( 2755166252928422419),  INT64_C( 2732664164436416460) },
      { -INT64_C( 3978579609577121253),  INT64_C( 6192942712613307700) } },
    { { -INT64_C( 6741449653343172108),  INT64_C( 1165091049881912532) },
      { -INT64_C( 5501487897388893719), -INT64_C( 6441578264795393311) },
      { -INT64_C( 6843599631116322617), -INT64_C( 8675794999610860015) } },
    { {  INT64_C( 5375087343847138995), -INT64_C( 1216671162808284439) },
      { -INT64_C( 3650340204444900419), -INT64_C(  860789784606255436) },
      {  INT64_C( 6107210603358008730), -INT64_C( 2864182961154121999) } },
    { { -INT64_C( 3494086435577085502),  INT64_C( 2036464169349077836) },
      {  INT64_C( 1573470821043731996), -INT64_C( 5214454829765945235) },
      { -INT64_C( 8166121044862030237), -INT64_C( 3820566619526995433) } },
    { {  INT64_C( 6557083888222440521), -INT64_C( 1096643188037395636) },
      {  INT64_C(  893065809763010542), -INT64_C( 5651071501970251801) },
      { -INT64_C( 4307482219990651907), -INT64_C( 6244421518502031391) } },
    { {  INT64_C( 4706790428297495528), -INT64_C( 2486069532575334725) },
      {  INT64_C( 3307771085309376315), -INT64_C( 1834762754833407633) },
      {  INT64_C( 3043341115843443976),  INT64_C( 7628288364956334847) } },
    { {  INT64_C(  384264238922557983),  INT64_C( 3600649845533327660) },
      {  INT64_C( 2593495818833699978), -INT64_C( 4433647995065792876) },
      { -INT64_C( 8664550425015011494),  INT64_C( 4905197484239704505) } },
    { { -INT64_C( 5747389706860603922),  INT64_C( 7967927551426023789) },
      {  INT64_C( 2719677821050920935),  INT64_C( 3145817382140229955) },
      {  INT64_C(  464342090749344343), -INT64_C( 5921538083712828883) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m128i r = simde_mm_rorv_epi64(a, b);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i a = simde_test_x86_random_i64x2();
    simde__m128i b = simde_test_x86_random_i64x2();
    simde__m128i r = simde_mm_rorv_epi64(a, b);

    simde_test_x86_write_i64x2(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_mask_rorv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t src[2];
    const simde__mmask8 k;
    const int64_t a[2];
    const int64_t b[2];
    const int64_t r[2];
  } test_vec[] = {
    { { -INT64_C( 5712674991527148950), -INT64_C( 8507325428741836261) },
      UINT8_C( 71),
      {  INT64_C( 7494458157836846923), -INT64_C( 6731928852836825039) },
      { -INT64_C(  449604565365469401), -INT64_C( 6624764979414144543) },
      { -INT64_C(  471811921691016365),  INT64_C( 1493686652924178736) } },
    { { -INT64_C( 3582789846887959325),  INT64_C( 5019695121268456615) },
      UINT8_C(217),
      {  INT64_C( 1745666772933143720),  INT64_C(  462486556078309996) },
      {  INT64_C( 5582336609227065970), -INT64_C( 3328809755821244567) },
      {  INT64_C( 8551093486821705230),  INT64_C( 5019695121268456615) } },
    { { -INT64_C(  764377699213267299),  INT64_C( 6664457642309440868) },
      UINT8_C( 50),
      {  INT64_C( 2808575625215028767),  INT64_C( 7550883709219499387) },
      {  INT64_C( 5697588418162267722),  INT64_C( 2557364009090778979) },
      { -INT64_C(  764377699213267299),  INT64_C( 5010814990317863959) } },
    { { -INT64_C( 3778328426953031485),  INT64_C( 5314180487252074396) },
      UINT8_C(216),
      { -INT64_C( 8144301397163968848),  INT64_C(   86613759766971610) },
      {  INT64_C(  915669079263649130),  INT64_C( 8840097448237665044) },
      { -INT64_C( 3778328426953031485),  INT64_C( 5314180487252074396) } },
    { {  INT64_C( 8926145049919488075), -INT64_C( 7850022463790681267) },
      UINT8_C(187),
      {  INT64_C( 7895360321318595680),  INT64_C( 1702274993141278975) },
      {  INT64_C( 4979068943017243074), -INT64_C( 7424182648954804193) },
      {  INT64_C( 1973840080329648920), -INT64_C( 6309875138254904276) } },
    { { -INT64_C( 3207505404878266194),  INT64_C( 2899411501643772942) },
      UINT8_C(185),
      {  INT64_C( 3911323560622945524),  INT64_C( 6696014252168021054) },
      {  INT64_C( 3300861972864630042),  INT64_C( 2714751593828578348) },
      {  INT64_C( 2521238494889902986),  INT64_C( 2899411501643772942) } },
    { {  INT64_C( 5040817743424468912), -INT64_C( 9011014115182419431) },
      UINT8_C(111),
      {  INT64_C(  527789848567300899),  INT64_C( 3032713599542314666) },
      {  INT64_C( 3123378047813866922),  INT64_C(  350741878965268220) },
      { -INT64_C( 8896304364021230395), -INT64_C( 6816814628451620190) } },
    { { -INT64_C( 8614965118453954728), -INT64_C( 5773047349728335420) },
      UINT8_C(217),
      {  INT64_C( 3857559887523606036), -INT64_C( 7075688460440897695) },
      {  INT64_C( 6019064933939624261),  INT64_C( 3877326746132496369) },
      { -INT64_C( 6796980281155969168), -INT64_C( 5773047349728335420) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi64(test_vec[i].src);
    simde__m128i a = simde_x_mm_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m128i r = simde_mm_mask_rorv_epi64(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i src = simde_test_x86_random_i64x2();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i64x2();
    simde__m128i b = simde_test_x86_random_i64x2();
    simde__m128i r = simde_mm_mask_rorv_epi64(src, k, a, b);

    simde_test_x86_write_i64x2(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_maskz_rorv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int64_t a[2];
    const int64_t b[2];
    const int64_t r[2];
  } test_vec[] = {
    { UINT8_C(247),
      {  INT64_C( 4657411723102175602),  INT64_C( 2371904861724022922) },
      {  INT64_C( 8440060390469709101),  INT64_C( 6374507612944602999) },
      { -INT64_C( 7329043287324359405), -INT64_C( 3069819662130670527) } },
    { UINT8_C(141),
      {  INT64_C( 6030907290051582132), -INT64_C( 1293841233286113749) },
      {  INT64_C( 8410921305400097828),  INT64_C( 7244236469531457839) },
      { -INT64_C( 6733609820966149789),  INT64_C(                   0) } },
    { UINT8_C( 75),
      { -INT64_C( 7737227872044741538), -INT64_C( 4210369181610836718) },
      { -INT64_C( 4130197174272994330), -INT64_C( 8074996799616210447) },
      { -INT64_C(  445269011394822304), -INT64_C( 2178415750161014072) } },
    { UINT8_C(101),
      {  INT64_C( 2332645401739592564),  INT64_C( 3510114308793682957) },
      { -INT64_C( 8351499022334347888),  INT64_C( 2321689913063226873) },
      {  INT64_C(  248859472756362659),  INT64_C(                   0) } },
    { UINT8_C( 79),
      { -INT64_C( 7412904314639585829), -INT64_C( 2541958015268491077) },
      { -INT64_C( 5233098128638287388),  INT64_C( 5090900925823005450) },
      { -INT64_C( 3368931262506860244),  INT64_C( 3384224538908327356) } },
    { UINT8_C(241),
      {  INT64_C( 7738687719068293784),  INT64_C( 2907054075131629133) },
      {  INT64_C( 5910368817374216110), -INT64_C( 7372106709453677764) },
      {  INT64_C( 6767409378285956501),  INT64_C(                   0) } },
    { UINT8_C( 94),
      {  INT64_C( 3896532494775239535), -INT64_C( 7065401391828793771) },
      {  INT64_C( 5683278789551648201), -INT64_C( 4670539230235090996) },
      {  INT64_C(                   0),  INT64_C( 7303113644083267560) } },
    { UINT8_C( 44),
      {  INT64_C( 2206776566129984775), -INT64_C( 4253199227850791799) },
      {  INT64_C( 1815237048546007894),  INT64_C( 3992766732294488660) },
      {  INT64_C(                   0),  INT64_C(                   0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_rorv_epi64(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x2(r, simde_x_mm_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i64x2();
    simde__m128i b = simde_test_x86_random_i64x2();
    simde__m128i r = simde_mm_maskz_rorv_epi64(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x2(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x2(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_rorv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[4];
    const int64_t b[4];
    const int64_t r[4];
  } test_vec[] = {
    { {  INT64_C( 1489263881975555604), -INT64_C( 8859794338791044252),  INT64_C( 1338588315493512040),  INT64_C( 2044618873627689261) },
      { -INT64_C( 6230452709232650280), -INT64_C(  627924979889020985), -INT64_C( 8076749209848724260),  INT64_C( 6198446769142992096) },
      { -INT64_C(  511699428660372427), -INT64_C( 3960327221319913578),  INT64_C( 6317202830687596465),  INT64_C( 6891423516812374536) } },
    { {  INT64_C( 1915247361442600870), -INT64_C( 5208368044738366926),  INT64_C( 4962535107812312300), -INT64_C( 4479056210683900669) },
      {  INT64_C( 2879841907145220489), -INT64_C( 1475877761633954062), -INT64_C( 8355558803135458716), -INT64_C(  810115309736478492) },
      { -INT64_C( 3238851014203939541),  INT64_C(  736039986982071790),  INT64_C( 3812556907614168950), -INT64_C( 5182837434230148319) } },
    { { -INT64_C( 6799933310642327930), -INT64_C( 8248356832462075490), -INT64_C( 4820152492936019707), -INT64_C( 7784333361530721895) },
      {  INT64_C( 3403788458291808748), -INT64_C( 6633710315684189639),  INT64_C( 4515506951436780820), -INT64_C( 3308109544327846026) },
      { -INT64_C( 8437585062955181540), -INT64_C( 4325262353701220538),  INT64_C( 6633903207377373429), -INT64_C( 2163922364932921777) } },
    { {  INT64_C( 1064443621759853828), -INT64_C(  462638320997854893), -INT64_C( 8464075612113942875),  INT64_C(  841932371045305591) },
      {  INT64_C( 1742444202192608331), -INT64_C( 6568607809245962504),  INT64_C( 4972388841520348515),  INT64_C( 9176032788336480069) },
      { -INT64_C( 6880980482759930460), -INT64_C( 7754945733193542663),  INT64_C( 7601188679114109442), -INT64_C( 5161836384135645593) } },
    { {  INT64_C( 6398433510807304685), -INT64_C( 3094239633904053140), -INT64_C( 4801314974531256374),  INT64_C( 4544237931753734236) },
      { -INT64_C( 1652049765603299239),  INT64_C( 2294975801412831377), -INT64_C( 3487872487055153821),  INT64_C( 6604653104398901526) },
      {  INT64_C( 5763752843431817305),  INT64_C( 7653421747076150683),  INT64_C(  218881950170728879),  INT64_C( 2738594277151033924) } },
    { { -INT64_C( 1691059502321460703), -INT64_C( 6637244543858320274), -INT64_C( 4094591535176344598),  INT64_C( 2239653503250238903) },
      { -INT64_C( 4777960421162342286),  INT64_C( 5550233964519726370),  INT64_C( 6050541685931935103),  INT64_C( 8179417629485442917) },
      {  INT64_C(  690712676934384162),  INT64_C( 3529475324913447248), -INT64_C( 8189183070352689195), -INT64_C(  735911457308760377) } },
    { { -INT64_C( 6244825241177407160), -INT64_C( 5100023959231080823),  INT64_C( 2343807750727394200), -INT64_C(  326222136307641848) },
      {  INT64_C( 2988393026311861063),  INT64_C( 3436332930276099261), -INT64_C( 9075622434897159887), -INT64_C( 2438414879195295620) },
      { -INT64_C( 7975123041368771454), -INT64_C( 3906703526429543347),  INT64_C( 8096796982389772355), -INT64_C( 5219554180922269553) } },
    { {  INT64_C( 7025092752680205494),  INT64_C( 1400630347164062633),  INT64_C( 4079712314524160930), -INT64_C( 2301180898677675876) },
      { -INT64_C( 8479797740871172788),  INT64_C( 6141849628217944392), -INT64_C( 2016750303311741938),  INT64_C( 5134991956473440824) },
      { -INT64_C( 3790315775710635317), -INT64_C( 6263539469006120813),  INT64_C( 5659021937669600255),  INT64_C( 1193500297220627680) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi64(test_vec[i].b);
    simde__m256i r = simde_mm256_rorv_epi64(a, b);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i a = simde_test_x86_random_i64x4();
    simde__m256i b = simde_test_x86_random_i64x4();
    simde__m256i r = simde_mm256_rorv_epi64(a, b);

    simde_test_x86_write_i64x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_mask_rorv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t src[4];
    const simde__mmask8 k;
    const int64_t a[4];
    const int64_t b[4];
    const int64_t r[4];
  } test_vec[] = {
    { {  INT64_C( 4006496326016145118), -INT64_C( 1006581310255347185), -INT64_C(  187369632019841945), -INT64_C( 7604050106431226227) },
      UINT8_C(147),
      {  INT64_C( 6598046757928695953),  INT64_C( 4104448579282417062), -INT64_C( 7885375976160891216), -INT64_C(  938472342692370327) },
      {  INT64_C( 6468299029619129048),  INT64_C( 6822351652379934776),  INT64_C( 2500048344083281098), -INT64_C( 5535062638384841763) },
      { -INT64_C( 4469662807704956543), -INT64_C(  725575905145674184), -INT64_C(  187369632019841945), -INT64_C( 7604050106431226227) } },
    { { -INT64_C( 2429726466379046341), -INT64_C( 3122120397410849120), -INT64_C( 2229102573981135052),  INT64_C( 6137366267807307934) },
      UINT8_C(240),
      { -INT64_C( 5131722539141807288), -INT64_C( 3243551863982647608), -INT64_C( 7868993638777820535), -INT64_C( 5110456992188840746) },
      { -INT64_C( 1690914602157429947),  INT64_C( 7662501900611242945), -INT64_C( 5034071224652190314),  INT64_C( 1716421012379062182) },
      { -INT64_C( 2429726466379046341), -INT64_C( 3122120397410849120), -INT64_C( 2229102573981135052),  INT64_C( 6137366267807307934) } },
    { {  INT64_C( 2208575115697304478), -INT64_C( 7419969546433177926), -INT64_C( 6429648234730644186),  INT64_C( 3748277568763962140) },
      UINT8_C( 65),
      { -INT64_C( 1290917256101891349),  INT64_C( 6160795339653755232), -INT64_C( 1000648704751466061), -INT64_C(  465914692405556808) },
      {  INT64_C( 4987291170352932783),  INT64_C( 7289390121472632603),  INT64_C( 3595985188286725798), -INT64_C( 2552976747717431414) },
      {  INT64_C( 8876796350614199339), -INT64_C( 7419969546433177926), -INT64_C( 6429648234730644186),  INT64_C( 3748277568763962140) } },
    { {  INT64_C( 7805400404050896424), -INT64_C( 5433478544518811518),  INT64_C(  689491090409651686),  INT64_C( 3677245214539450879) },
      UINT8_C( 26),
      { -INT64_C( 9004711525823659988), -INT64_C( 3286143574138078186),  INT64_C( 6880754029418300654), -INT64_C( 2501577308202489349) },
      { -INT64_C( 6921068864786353187), -INT64_C( 5237822940197780312), -INT64_C(  793561549658284217), -INT64_C( 2311636327153375908) },
      {  INT64_C( 7805400404050896424), -INT64_C( 3134908603336334011),  INT64_C(  689491090409651686), -INT64_C( 1829974660909249774) } },
    { {  INT64_C( 7225817505167527710),  INT64_C( 3407026893527016903), -INT64_C( 8233204431893791280),  INT64_C( 7498359557796388654) },
      UINT8_C(214),
      { -INT64_C( 4870653615761387426), -INT64_C( 4766827933595273000),  INT64_C( 4509927722275742112),  INT64_C( 3707291343035876886) },
      { -INT64_C( 3236896281459424255),  INT64_C( 3112467328526956595), -INT64_C( 5696443734727319723),  INT64_C( 7852799369883187953) },
      {  INT64_C( 7225817505167527710),  INT64_C( 1902772030644361147),  INT64_C( 8434399827635925831),  INT64_C( 7498359557796388654) } },
    { {  INT64_C( 4436564644303523523),  INT64_C( 8523028352153447824), -INT64_C(  627658383382881536),  INT64_C( 1832885153404131223) },
      UINT8_C(133),
      { -INT64_C( 2982284101671725005), -INT64_C( 2639551383720183684), -INT64_C( 2716558689885357672), -INT64_C( 1021390513578684778) },
      { -INT64_C( 6169200114347974110),  INT64_C( 6930309507365493074),  INT64_C( 9042772749307940704),  INT64_C( 1900130882628071087) },
      {  INT64_C( 4837980960247460634),  INT64_C( 8523028352153447824), -INT64_C( 5340713248191883197),  INT64_C( 1832885153404131223) } },
    { {  INT64_C( 5104840022472153856), -INT64_C( 6302288190290468511), -INT64_C( 5892175062247956779), -INT64_C( 6662097480897067656) },
      UINT8_C(212),
      { -INT64_C( 8459431730568638159), -INT64_C( 1566273776699391064),  INT64_C( 4469708528183541702),  INT64_C( 3747421514138482867) },
      { -INT64_C( 9040686139799909803), -INT64_C( 7205139459221726005),  INT64_C( 4011075057675463474),  INT64_C( 7413599058924965035) },
      {  INT64_C( 5104840022472153856), -INT64_C( 6302288190290468511), -INT64_C( 1869446867772665983), -INT64_C( 6662097480897067656) } },
    { {  INT64_C( 2489626751633018672), -INT64_C( 2180521021214720950), -INT64_C( 3971620590069094105), -INT64_C( 2595663641760545404) },
      UINT8_C(169),
      { -INT64_C( 1326272220225280673),  INT64_C( 6971147225814317403), -INT64_C( 8122446116018100021),  INT64_C( 3872838213802056637) },
      {  INT64_C( 4420314788761157626),  INT64_C( 4444369426897499349),  INT64_C( 3302342974272187130),  INT64_C( 2174679127385443874) },
      {  INT64_C( 7352298274129795067), -INT64_C( 2180521021214720950), -INT64_C( 3971620590069094105), -INT64_C( 8822913481059613323) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi64(test_vec[i].src);
    simde__mmask8 k = test_vec[i].k;
    simde__m256i a = simde_x_mm256_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi64(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_rorv_epi64(src, k, a, b);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i src = simde_test_x86_random_i64x4();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m256i a = simde_test_x86_random_i64x4();
    simde__m256i b = simde_test_x86_random_i64x4();
    simde__m256i r = simde_mm256_mask_rorv_epi64(src, k, a, b);

    simde_test_x86_write_i64x4(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_maskz_rorv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int64_t a[4];
    const int64_t b[4];
    const int64_t r[4];
  } test_vec[] = {
    { UINT8_C(195),
      { -INT64_C( 2908232570122237263), -INT64_C( 4088557505929941067),  INT64_C( 7633431334080710278), -INT64_C( 4075851023874398039) },
      { -INT64_C( 6057560348459461579),  INT64_C( 4939099410294519847), -INT64_C( 3423298498897788229), -INT64_C( 1385159560377070616) },
      {  INT64_C( 2238032197843259069),  INT64_C( 4075357300608369933),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(  0),
      {  INT64_C( 3151008956830581851), -INT64_C( 3710072147974455661),  INT64_C( 2461151325517486142),  INT64_C( 5129255743305360961) },
      {  INT64_C( 6406815521942676682),  INT64_C( 4605109839853572883), -INT64_C( 2200670734886446020), -INT64_C( 2306339895203967029) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C( 96),
      {  INT64_C( 6521351380549344728),  INT64_C( 7025221266367487134), -INT64_C( 7671633171779038953),  INT64_C( 1697164634247282928) },
      { -INT64_C( 3528565696616942809),  INT64_C( 1479027840754213792),  INT64_C( 6502170272338118496), -INT64_C( 7426694781314013287) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(199),
      { -INT64_C( 5613798166853302413),  INT64_C( 5461111267222568409), -INT64_C( 7744204894060121964), -INT64_C( 4507648790835667051) },
      {  INT64_C(  857459856103729892), -INT64_C( 1397454710340545140), -INT64_C(  869369145163362386),  INT64_C( 8561078536639476928) },
      { -INT64_C(  612625376671073148), -INT64_C( 2480150115563169037),  INT64_C( 4231051386962465308),  INT64_C(                   0) } },
    { UINT8_C(121),
      {  INT64_C( 2802284128495184391),  INT64_C( 2096262685841634091),  INT64_C( 6297843598414803781),  INT64_C( 5374024997195576332) },
      {  INT64_C( 2432371468464635658), -INT64_C( 1035087201994915657), -INT64_C( 7363668457153880556),  INT64_C( 2306204846093875059) },
      { -INT64_C( 9094534641694168342),  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 8365326918538393262) } },
    { UINT8_C(137),
      { -INT64_C( 4765476371310855262),  INT64_C( 2513108668223414826), -INT64_C( 2565845460376922125), -INT64_C( 4697233399160366082) },
      {  INT64_C( 1804186359253192654), -INT64_C( 4590639385222483421),  INT64_C( 4619873266577897756), -INT64_C( 1077670781472207055) },
      { -INT64_C( 1257921057538139650),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(  688527345622933352) } },
    { UINT8_C(168),
      { -INT64_C( 5128560637771301649),  INT64_C( 4501143616282295190), -INT64_C( 1999385113736922888), -INT64_C( 8625808797542675127) },
      {  INT64_C( 2896391333608022324), -INT64_C(  753232546837055737),  INT64_C( 1687799609881267378), -INT64_C( 6921184442451397679) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 4132962892741029629) } },
    { UINT8_C(228),
      {  INT64_C(  234362326041958700),  INT64_C( 2580833165614550051),  INT64_C( 3895341837366417402), -INT64_C( 7374572450113029439) },
      { -INT64_C( 2114139035238588582), -INT64_C( 7360623831100184162), -INT64_C( 7844734182833534425),  INT64_C( 6802686153978266209) },
      {  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 8755592855190495727),  INT64_C(                   0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__mmask8 k = test_vec[i].k;
    simde__m256i a = simde_x_mm256_loadu_epi64(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi64(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_rorv_epi64(k, a, b);
    simde_test_x86_assert_equal_i64x4(r, simde_x_mm256_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m256i a = simde_test_x86_random_i64x4();
    simde__m256i b = simde_test_x86_random_i64x4();
    simde__m256i r = simde_mm256_maskz_rorv_epi64(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x4(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_rorv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { { -INT64_C( 4031222765621492941), -INT64_C( 4450970416159848894), -INT64_C( 1625311065313362393),  INT64_C( 1326347335033401858),
        -INT64_C( 8363776830626917916),  INT64_C( 1345420615588967147), -INT64_C( 1856316595130324958), -INT64_C( 5283558186717040453) },
      {  INT64_C( 6191545042178114584), -INT64_C( 7081372226062215493),  INT64_C( 1719019754786738705), -INT64_C( 7392554877203678497),
        -INT64_C( 3989654854368082974), -INT64_C( 2696485433278210180), -INT64_C( 7715321793429404294), -INT64_C( 2359694899797451227) },
      { -INT64_C( 6214066126516109116),  INT64_C( 5142899272561248344),  INT64_C( 2239402514570625854),  INT64_C( 8279696408904221631),
         INT64_C( 5293352961600878930),  INT64_C( 3079985775713922737), -INT64_C( 8123797646083487559), -INT64_C( 7722268709641426882) } },
    { { -INT64_C( 7005969198951482383), -INT64_C( 2932360826006981461), -INT64_C( 1227342187978744402), -INT64_C( 2659984802977596359),
        -INT64_C( 5057368602012270320),  INT64_C( 4325561817719727570), -INT64_C( 8624509493706771304), -INT64_C( 3848093086882274093) },
      { -INT64_C( 2171918034187554796),  INT64_C( 2016231112792082481),  INT64_C(  334334209130535884),  INT64_C( 5274657670228435375),
         INT64_C( 3222685055841939402), -INT64_C( 5914098353080382898), -INT64_C( 1965984838225204863), -INT64_C( 7184788616062862900) },
      { -INT64_C( 5494644517409513518),  INT64_C( 1490333356285881255), -INT64_C( 2670934223119566127), -INT64_C( 6065102768984312277),
         INT64_C( 4912991956563179024), -INT64_C( 7545517164317852491),  INT64_C( 4911117290001390156),  INT64_C( 5565509663578643438) } },
    { {  INT64_C( 5320491668216329647),  INT64_C( 2992367795399475704), -INT64_C( 8608468253584633745),  INT64_C(  323084164063155468),
         INT64_C( 1714890829009449096),  INT64_C( 6635343684494812741), -INT64_C( 7994865131462790711), -INT64_C( 1101433124431030947) },
      {  INT64_C( 3248010026147865820),  INT64_C( 5075046044314174501),  INT64_C( 5996498883210604064),  INT64_C( 8126388821245142893),
        -INT64_C( 7046868458647373537), -INT64_C( 7113683568472076925),  INT64_C( 6257041301120881292), -INT64_C( 5465977322934052319) },
      { -INT64_C( 6387411639835889734),  INT64_C( 4526352988663199797), -INT64_C( 6567285716650781765), -INT64_C( 7300620531158473762),
         INT64_C(  530827790733871167), -INT64_C( 6088111067079230264),  INT64_C( 2060696757965701135),  INT64_C( 6428970583422891572) } },
    { { -INT64_C( 7979271268598050773), -INT64_C( 7037198693273767107),  INT64_C( 3687311280661699051),  INT64_C( 8366294242169477073),
         INT64_C(  233163289060300444), -INT64_C( 7926313734551843249), -INT64_C( 1661104681869958740), -INT64_C( 8960356871687906583) },
      {  INT64_C( 7712812501149073216), -INT64_C( 8830489208191924679),  INT64_C( 6191663800765373165), -INT64_C( 1767331183642330097),
         INT64_C( 5364903233664690157), -INT64_C(  424571962960166723), -INT64_C( 6685052047613534600), -INT64_C( 2686939315080413143) },
      { -INT64_C( 7979271268598050773),  INT64_C( 3129026872725839567), -INT64_C( 5722209200137201313), -INT64_C( 5790810851806884400),
        -INT64_C( 1858481626399368734), -INT64_C( 8070277655286091140), -INT64_C(  967684863389750040),  INT64_C( 8293704089599071029) } },
    { {  INT64_C( 6347074476989195449),  INT64_C( 6241155510443065494), -INT64_C( 5647668375531643801),  INT64_C(  358941391130277939),
        -INT64_C( 3760816621018423589),  INT64_C( 3267937420421544997),  INT64_C( 4387006350017205671),  INT64_C( 8453072445310329344) },
      {  INT64_C( 2970047007444180791),  INT64_C( 6442831036730418807),  INT64_C( 6440545873230926365),  INT64_C( 5682685211194998477),
        -INT64_C(  414065620376958957),  INT64_C( 1414870212598460055),  INT64_C( 7126271952460196658), -INT64_C( 8017068703973013160) },
      {  INT64_C( 3075175245586985648),  INT64_C( 4184896595097103533), -INT64_C( 2750323618918720441), -INT64_C( 4496800411841655799),
        -INT64_C(  298511385990091047), -INT64_C( 2488157040693245432),  INT64_C( 8197127509484621624),  INT64_C( 6728941297087322699) } },
    { {  INT64_C(   66402161250498414), -INT64_C(  837691785931578794), -INT64_C( 8498745282291329258), -INT64_C( 1370018433022174994),
        -INT64_C( 2190779617330158802), -INT64_C( 2044197940847844885),  INT64_C( 8349566529671471396), -INT64_C( 4692509396089972362) },
      {  INT64_C( 8353032201945136766), -INT64_C( 3926924858522265137),  INT64_C( 2584220306950262755),  INT64_C( 1524178848769883095),
         INT64_C( 4157396857441451190),  INT64_C( 8760159426525451043), -INT64_C( 8703559566389759643), -INT64_C(  277792411337404806) },
      {  INT64_C(  265608645001993656),  INT64_C( 4948867476202231954),  INT64_C( 3859570461667085446), -INT64_C(  886683723352834857),
         INT64_C( 7144448846482684806), -INT64_C( 5357669209420320359),  INT64_C( 3074965725746494708), -INT64_C( 5172696170405405265) } },
    { {  INT64_C( 8229463346885903505), -INT64_C( 4121129695855310144), -INT64_C( 1605797771915710337),  INT64_C( 5521448033926620765),
        -INT64_C( 8636360608498328733), -INT64_C( 5860119697674116986), -INT64_C(  957977449631742628),  INT64_C( 3477120645523222074) },
      { -INT64_C( 8475484206806104047),  INT64_C( 5135543843486187112), -INT64_C( 1068822371830193502),  INT64_C(  536536343760060533),
        -INT64_C( 4161535659927906895),  INT64_C( 8048393007929339775), -INT64_C(   41147528377875516),  INT64_C( 3282478150344332432) },
      { -INT64_C( 6752944117931049571),  INT64_C( 6941519318031453902), -INT64_C( 3749690755138075685),  INT64_C(   71456297764186725),
        -INT64_C( 4763584495004564462),  INT64_C( 6726504678361317645), -INT64_C( 3518638104422524843), -INT64_C( 5892344105810397802) } },
    { {  INT64_C( 3327352732647582985), -INT64_C( 1149463348914910511),  INT64_C( 1483612286469583628), -INT64_C( 7649600857459468243),
        -INT64_C( 6020703730108941931),  INT64_C( 6904113146987350707),  INT64_C( 4800992326475725833), -INT64_C(  538529370907919515) },
      {  INT64_C( 6918565507792405372), -INT64_C( 6094760482068017992),  INT64_C( 6026617769703268683), -INT64_C( 3715527621115343341),
         INT64_C( 1322411612772215630), -INT64_C(  738157504615245197), -INT64_C( 1932896559870641686), -INT64_C( 8456693312628493946) },
      { -INT64_C( 2102588498767327086),  INT64_C(  885287857135735280),  INT64_C( 7026339838759726486),  INT64_C( 6558844948111791411),
        -INT64_C( 1849095121628490766),  INT64_C(  777570126891740154),  INT64_C( 4999907695739840547), -INT64_C( 7502404301364941587) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_rorv_epi64(a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m512i b = simde_test_x86_random_i64x8();
    simde__m512i r = simde_mm512_rorv_epi64(a, b);

    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_mask_rorv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t src[8];
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { {  INT64_C( 5237100477281721135), -INT64_C( 6081791155903194878), -INT64_C( 3450607602809573641),  INT64_C( 4884474943074533144),
         INT64_C( 2120053148466032640), -INT64_C( 3611796920684989686),  INT64_C( 4015452885585569087),  INT64_C( 2168738439701199553) },
      UINT8_C( 24),
      {  INT64_C( 6942670105351569757), -INT64_C( 7811208306202604023),  INT64_C( 3903469947744060072),  INT64_C( 8433679803169898505),
        -INT64_C( 7367575013743640908), -INT64_C( 6652786439921338453),  INT64_C( 5800979673882910223), -INT64_C(  160356858945415624) },
      {  INT64_C( 5958861159021906570),  INT64_C( 4807055160401176088),  INT64_C( 3822729236893844597),  INT64_C( 2678600107762798565),
        -INT64_C( 4336419247321281577), -INT64_C( 3959353102068990825), -INT64_C( 8033695326109525167),  INT64_C( 3216953686785657059) },
      {  INT64_C( 5237100477281721135), -INT64_C( 6081791155903194878), -INT64_C( 3450607602809573641), -INT64_C( 4068263390246055016),
        -INT64_C( 8611611242513359165), -INT64_C( 3611796920684989686),  INT64_C( 4015452885585569087),  INT64_C( 2168738439701199553) } },
    { {  INT64_C( 7846532606109344629), -INT64_C( 7361902939633894472), -INT64_C( 3235438375462347553),  INT64_C( 4103614626410977393),
        -INT64_C( 3450816038242247206),  INT64_C( 8564072529119695864), -INT64_C( 7291018653573595122), -INT64_C(  726395536451492623) },
      UINT8_C(240),
      {  INT64_C( 1475731253050844484),  INT64_C( 6086088867732765622),  INT64_C( 7577896008600613920),  INT64_C( 1062140094267580025),
         INT64_C( 6193587069324936735), -INT64_C( 2256809462016925755), -INT64_C( 4803576889717674841),  INT64_C( 8429126987431926562) },
      {  INT64_C( 6485544741773806092), -INT64_C( 7311766675677462553),  INT64_C(  616955657417367950),  INT64_C( 7818792520115296696),
         INT64_C(  317249911317386616),  INT64_C(  893176716483828696), -INT64_C( 9024247415617688364),  INT64_C( 2184828177758533511) },
      {  INT64_C( 7846532606109344629), -INT64_C( 7361902939633894472), -INT64_C( 3235438375462347553),  INT64_C( 4103614626410977393),
        -INT64_C(  861700591837634731), -INT64_C( 4782886710458204070), -INT64_C( 6716419737829224751),  INT64_C( 4965768949168411574) } },
    { {  INT64_C( 6631711547006348006), -INT64_C( 4956337940408585043),  INT64_C(  775808271733101944),  INT64_C( 5455340622937412592),
         INT64_C( 2345908608183558318), -INT64_C( 4922854118624872949),  INT64_C( 5559343395681150104),  INT64_C(  631798461609809684) },
      UINT8_C(121),
      {  INT64_C( 9184543648802159123),  INT64_C( 4211846514139555928), -INT64_C( 3796563398610153863), -INT64_C( 6927892421302368055),
         INT64_C( 3292553356228806761),  INT64_C( 7510170124306379496),  INT64_C( 9118367257966399498),  INT64_C( 5180754609217515706) },
      {  INT64_C( 6234717387576962692),  INT64_C( 1509986851519500189),  INT64_C( 4869056990465464158), -INT64_C( 5606015963346931187),
         INT64_C( 8368131038153256029),  INT64_C( 7007869314632021281),  INT64_C( 9032257606134272908), -INT64_C( 5883219188202806663) },
      {  INT64_C( 4032798491870675873), -INT64_C( 4956337940408585043),  INT64_C(  775808271733101944),  INT64_C( 5065703890798973238),
        -INT64_C( 8048421501414932823), -INT64_C(   70868522877796873),  INT64_C(   47262160155044413),  INT64_C(  631798461609809684) } },
    { {  INT64_C( 1770929464662385992),  INT64_C( 1401609624704291599),  INT64_C( 7459230841666398359), -INT64_C( 3430782455251997596),
        -INT64_C( 8728395581142082354),  INT64_C( 4445066741945999893), -INT64_C( 4662694167224985100), -INT64_C( 8066553028784334411) },
      UINT8_C(181),
      {  INT64_C( 6310582346184349436), -INT64_C( 5167251792030978230), -INT64_C( 3459376433781855104),  INT64_C( 4887850659464952392),
        -INT64_C(  286729874872139400), -INT64_C( 4747748115387120764),  INT64_C( 5336488491151208393),  INT64_C( 4308327197756334143) },
      {  INT64_C( 7273574948640688961), -INT64_C( 6274108158138432528),  INT64_C( 4062519513074693139),  INT64_C( 5590524094798810007),
         INT64_C( 6582065195131174164),  INT64_C( 1656330718664073076),  INT64_C( 8330968660116759660), -INT64_C(  527093898329367466) },
      {  INT64_C( 3155291173092174718),  INT64_C( 1401609624704291599),  INT64_C( 7318377980612233474), -INT64_C( 3430782455251997596),
         INT64_C( 1159553485298674396), -INT64_C( 3908026935779243039), -INT64_C( 4662694167224985100), -INT64_C( 5872415809975678439) } },
    { { -INT64_C( 7591305212104740641),  INT64_C( 1548647457857613924), -INT64_C( 8495732607647307573),  INT64_C( 2905273969662551668),
        -INT64_C( 1854611672011752053), -INT64_C( 2448047160427729591),  INT64_C(  950715379112837490),  INT64_C( 4969078673004371333) },
      UINT8_C(244),
      {  INT64_C( 7046264472561185128),  INT64_C( 4820351066500873129), -INT64_C( 2667000978852836136),  INT64_C( 2880290276630579534),
         INT64_C( 8313488553818943177),  INT64_C(   13735275044197420),  INT64_C( 4823201218106428248),  INT64_C( 2603994673857332029) },
      {  INT64_C( 7488221997363808941), -INT64_C(  684681702566145450), -INT64_C( 6357118488598022512), -INT64_C(  432971955803320963),
        -INT64_C(  562002578990096496), -INT64_C( 7006487155627882449),  INT64_C( 5115501798410604268), -INT64_C(  703938790743432970) },
      { -INT64_C( 7591305212104740641),  INT64_C( 1548647457857613924), -INT64_C( 8297641533658794021),  INT64_C( 2905273969662551668),
         INT64_C( 7983038668259131047), -INT64_C( 7470948630491824031),  INT64_C( 4558020439468945143), -INT64_C( 8287344657976986480) } },
    { {  INT64_C( 4167226081738403131),  INT64_C( 3182577365754989438),  INT64_C( 3145059782000528535),  INT64_C( 1908587119579612191),
        -INT64_C( 5505145643658166698), -INT64_C(   35068475227529450), -INT64_C( 6551043634737838411), -INT64_C( 8952410394250804024) },
      UINT8_C(229),
      {  INT64_C( 1989408495578506169),  INT64_C( 8875049456664572427), -INT64_C( 5788068713638849056), -INT64_C( 3504821859896240944),
         INT64_C( 5609776781295056194), -INT64_C( 4624654902650315552),  INT64_C( 5322217336720314881),  INT64_C( 6523776427760859300) },
      { -INT64_C( 6115339861758452591), -INT64_C( 3392103532550036657), -INT64_C( 8905911280997723995), -INT64_C( 3169470619322538095),
         INT64_C( 6780953257077906595), -INT64_C( 7760976567172654164), -INT64_C( 5662123002377910520),  INT64_C( 1465742945848830528) },
      { -INT64_C( 1018783500221922804),  INT64_C( 3182577365754989438),  INT64_C( 6854488180662297905),  INT64_C( 1908587119579612191),
        -INT64_C( 5505145643658166698), -INT64_C( 3610360616641102562),  INT64_C(   92847505509491666),  INT64_C( 6523776427760859300) } },
    { {  INT64_C( 8423875112645951354), -INT64_C(  992134513390816584),  INT64_C( 5067471000037944706), -INT64_C( 8901892384488837648),
         INT64_C( 5646938436767971616),  INT64_C(  667712002511424615),  INT64_C( 6154730076122062365),  INT64_C( 8302998700565148331) },
      UINT8_C( 38),
      {  INT64_C(  406180177802713927),  INT64_C( 6291129798048412672),  INT64_C( 1002352025414693184),  INT64_C( 2313675762368028360),
        -INT64_C( 2095018376624810193),  INT64_C( 5585413572559796467),  INT64_C( 1853922086474170322),  INT64_C( 2092813657477880595) },
      {  INT64_C( 6540899458023253761),  INT64_C(  776036648919423082), -INT64_C( 2832356497643057080), -INT64_C( 1469915197354033011),
        -INT64_C(  434715494469099978), -INT64_C( 3553561629042836116),  INT64_C( 7359753537605494662), -INT64_C( 1392321784415393832) },
      {  INT64_C( 8423875112645951354),  INT64_C( 5623908779580314533),  INT64_C( 4615601456026664049), -INT64_C( 8901892384488837648),
         INT64_C( 5646938436767971616),  INT64_C(   59322120759728182),  INT64_C( 6154730076122062365),  INT64_C( 8302998700565148331) } },
    { {  INT64_C( 9052657539321789464), -INT64_C( 3507791425865379075), -INT64_C( 6307222214754518983), -INT64_C( 1385112157411452635),
        -INT64_C( 3521191684596280940),  INT64_C( 5791809541927748936), -INT64_C( 2462102611711329700), -INT64_C( 1588503665864851917) },
      UINT8_C(209),
      { -INT64_C(  718480607320830328),  INT64_C( 8269838137238067396),  INT64_C(  201545723801235647),  INT64_C( 5769746400960566465),
        -INT64_C( 1552028666969475889), -INT64_C( 6982806068119043154),  INT64_C(  994175445110375174),  INT64_C( 3814936344678769625) },
      { -INT64_C( 4313275168639942636), -INT64_C( 7913242855518603602),  INT64_C( 4980744157524682694), -INT64_C( 2097314248011649716),
         INT64_C( 8265022055219239384),  INT64_C( 7482221295816482231), -INT64_C( 6392691054873070697), -INT64_C( 2470721349831086586) },
      {  INT64_C( 2947763600591887338), -INT64_C( 3507791425865379075), -INT64_C( 6307222214754518983), -INT64_C( 1385112157411452635),
         INT64_C( 1131758012287297152),  INT64_C( 5791809541927748936),  INT64_C( 8108181491781638548),  INT64_C( 7265367784178399375) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi64(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_rorv_epi64(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i64x8();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m512i b = simde_test_x86_random_i64x8();
    simde__m512i r = simde_mm512_mask_rorv_epi64(src, k, a, b);

    simde_test_x86_write_i64x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_rorv_epi64 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { UINT8_C(213),
      { -INT64_C( 7148606243591303982),  INT64_C( 1789236256454807309),  INT64_C( 5601663334537811741),  INT64_C( 1838929107309793651),
         INT64_C( 4042253531198857212),  INT64_C( 1586756880992457991),  INT64_C(  616864043336825122),  INT64_C( 1115552481879319067) },
      { -INT64_C( 6358903863494662052), -INT64_C( 6771274735657142537),  INT64_C( 8768072374901068818),  INT64_C(   33309311368914517),
        -INT64_C( 7257752406890905089), -INT64_C( 8745547970192352386),  INT64_C( 5565846575989425510),  INT64_C( 7991122348410673217) },
      {  INT64_C( 4151418535940245489),  INT64_C(                   0),  INT64_C( 3875157741769584029),  INT64_C(                   0),
         INT64_C( 8084507062397714424),  INT64_C(                   0),  INT64_C( 5838694697561898538), -INT64_C( 8665595795915116275) } },
    { UINT8_C( 44),
      {  INT64_C( 3401460530648386926),  INT64_C( 6408386783278740214),  INT64_C(  680780152926417750), -INT64_C( 6005015472054445998),
        -INT64_C( 2776303267357755148), -INT64_C( 7209048843606746168),  INT64_C( 3818558678564095236),  INT64_C( 5872793273137985430) },
      {  INT64_C( 4724014843803919059),  INT64_C( 4975863242344493101), -INT64_C( 5895994657246357226),  INT64_C(  891153319840702383),
         INT64_C( 4080068892402317542),  INT64_C( 7262257046490872898), -INT64_C( 1560344186818067823), -INT64_C( 4273821631652570262) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 4813600496062809285), -INT64_C( 3711816081197409965),
         INT64_C(                   0),  INT64_C( 2809423807525701362),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C( 78),
      {  INT64_C( 4541476440441889650),  INT64_C( 6018200336354609467),  INT64_C( 7008858341171342927),  INT64_C(  725647539839166318),
         INT64_C( 3902603255541033511), -INT64_C(  720103529709456035),  INT64_C( 8413309071524925147),  INT64_C( 3286864695150510843) },
      { -INT64_C( 7140950616379593803), -INT64_C( 7453092634890203148),  INT64_C( 8166359238784410073), -INT64_C( 8290667490933699394),
         INT64_C( 4316456453163190028),  INT64_C( 6162446550901862705), -INT64_C(  454712674807873591),  INT64_C( 7023555302951597077) },
      {  INT64_C(                   0),  INT64_C( 5698495232519419192),  INT64_C( 5086577943727817581),  INT64_C( 2902590159356665272),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C( 7906738791433431111),  INT64_C(                   0) } },
    { UINT8_C(181),
      { -INT64_C( 4321424209332097868), -INT64_C(  486505881546984179), -INT64_C( 2132123271595524406), -INT64_C( 9057133787511040534),
        -INT64_C( 8170830988153632188),  INT64_C( 4308048670361967445),  INT64_C( 1644367107041722049), -INT64_C( 1973143297129052083) },
      { -INT64_C( 6028536116175702458),  INT64_C( 6835415977274661425), -INT64_C( 5799195537046672572),  INT64_C( 8622818995257095206),
        -INT64_C( 1413403799120353954),  INT64_C( 7559690154710785774),  INT64_C( 8856149808542922453),  INT64_C( 8663212754577961361) },
      { -INT64_C( 3238056390939643214),  INT64_C(                   0), -INT64_C( 5897865227508955156),  INT64_C(                   0),
         INT64_C( 6729830376441479687),  INT64_C(  991706795134480165),  INT64_C(                   0),  INT64_C(  299192846354746839) } },
    { UINT8_C( 79),
      {  INT64_C( 7996975182592486775), -INT64_C( 6430019384792414588), -INT64_C( 7799541994602431789),  INT64_C( 3814307806269179061),
         INT64_C( 8380983337317104210), -INT64_C( 5704330246088367423), -INT64_C(  318451252073541344), -INT64_C( 5253295695601123034) },
      {  INT64_C( 6118958763767060613), -INT64_C( 7901746473203315510),  INT64_C(  103064760166332460),  INT64_C( 1839435213883836777),
        -INT64_C( 3297346277291909291),  INT64_C( 5389068352540492485), -INT64_C( 1232813643452883703),  INT64_C( 1937184206966540596) },
      { -INT64_C( 4938241296274796181), -INT64_C( 6833736350899133279),  INT64_C( 6778779111322631207),  INT64_C( 6832055477708945298),
         INT64_C(                   0),  INT64_C(                   0), -INT64_C( 8035043710330671000),  INT64_C(                   0) } },
    { UINT8_C( 97),
      {  INT64_C(  193438458996917105),  INT64_C(  976834376054597556),  INT64_C( 2271081220308890150), -INT64_C( 6853916578406904515),
         INT64_C( 4126486403088009730), -INT64_C(  874175731280681952),  INT64_C( 3329788437296607679),  INT64_C( 2081946270736194069) },
      { -INT64_C( 3992542601107509543),  INT64_C( 7190129454450562567), -INT64_C( 1582540461015757591),  INT64_C( 2233806610355959753),
         INT64_C(  788714884278607746), -INT64_C( 1336822503487599509),  INT64_C( 7238358392564608180),  INT64_C( 4101733310593016455) },
      {  INT64_C( 3783508027628167285),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C(                   0), -INT64_C( 5063673342056301623),  INT64_C( 6669568695546409699),  INT64_C(                   0) } },
    { UINT8_C(167),
      { -INT64_C( 2240062164156456343),  INT64_C( 6264174385903761554), -INT64_C( 8218692280174676071),  INT64_C( 1480975830707584670),
         INT64_C( 8940344456068085469),  INT64_C( 7004404503104910963),  INT64_C( 1915080407926842907),  INT64_C( 3330848779069871738) },
      { -INT64_C( 4960027486929826621), -INT64_C( 6876489184088301464), -INT64_C( 7959768093580436493), -INT64_C( 2697956247357454065),
        -INT64_C( 4053607592947618117),  INT64_C( 1781041866250606482), -INT64_C( 6373700028178407708), -INT64_C( 1329804124790242662) },
      {  INT64_C( 4331678247907830861),  INT64_C( 7654682272927772369),  INT64_C( 3088709848917029310),  INT64_C(                   0),
         INT64_C(                   0), -INT64_C( 4783710878259071059),  INT64_C(                   0),  INT64_C( 4626778514419569737) } },
    { UINT8_C(132),
      {  INT64_C(  384683585302687762), -INT64_C( 9074963100529496670), -INT64_C( 2849439810570286391), -INT64_C( 4331628654224661048),
         INT64_C( 1412403438050330342), -INT64_C( 3607017145122241718), -INT64_C( 7726737919198164137), -INT64_C( 6001655882239979711) },
      {  INT64_C( 7142948810124180635),  INT64_C(  247630551233683101),  INT64_C( 8371829085775664845), -INT64_C( 7856780280283651707),
        -INT64_C( 5849122508911003817),  INT64_C( 3804626237701268509), -INT64_C(  469222685859201579),  INT64_C( 6099347228479250973) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 6219123253390528720),  INT64_C(                   0),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 7521440129703039441) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_rorv_epi64(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m512i b = simde_test_x86_random_i64x8();
    simde__m512i r = simde_mm512_maskz_rorv_epi64(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_rorv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_rorv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_rorv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_rorv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_rorv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_rorv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_rorv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_rorv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_rorv_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_rorv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_rorv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_rorv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_rorv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_rorv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_rorv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_rorv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_rorv_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_rorv_epi64)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
